# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import random
import six
import threading
import time

from logging import getLogger
from retrying import retry
from redis.exceptions import (ConnectionError, ResponseError, TimeoutError)

log = getLogger(__name__)


@six.python_2_unicode_compatible
class HealthError(Exception):
    def __init__(self, message):
        self.message = message

    def __str__(self):
        return '<{} message={}/>'.format(
            self.__class__.__name__,
            self.message
        )


class RedisServerError(HealthError):
    pass


class SharedCacheServerError(HealthError):
    pass


class RedisBackgroundPing(object):
    AVERAGE_REPLICATION_LAG = 1

    def __init__(self, cache, ping_key, ttl, sleep_time, maximum_lag, readiness_flag):
        self._cache = cache
        self._ping_key = ping_key
        self._ttl = ttl
        self._sleep_time = sleep_time
        self._thr = threading.Thread(target=self._ping)
        self._alive = False
        self._maximum_lag = maximum_lag
        self._readiness_flag = readiness_flag

    def run(self):
        self._alive = True
        if not self._thr.is_alive():
            self._thr.start()

    def stop(self):
        self._alive = False
        if self._thr.is_alive():
            self._thr.join()

    def _ping(self):
        while self._alive:
            try:
                timestamp = int(time.time() * 1000.0)
                self._set_ping_key(timestamp)
                if self._maximum_lag > 0:
                    time.sleep(self.AVERAGE_REPLICATION_LAG)
                if self._check_ping_key(timestamp):
                    self._readiness_flag.set()
            except Exception:
                log.exception('Ping error')
                self._readiness_flag.reset()
            time.sleep(self._sleep_time / 1000)

    @retry(
        wait_exponential_multiplier=200,
        wait_exponential_max=5000,
        stop_max_attempt_number=5,
        retry_on_exception=lambda e: isinstance(e, (ConnectionError, ResponseError, TimeoutError)),
    )
    def _set_ping_key(self, timestamp):
        self._cache.set(self._ping_key, timestamp, self._ttl)

    @retry(
        wait_exponential_multiplier=200,
        wait_exponential_max=5000,
        stop_max_attempt_number=5,
        retry_on_exception=lambda e: isinstance(e, (ConnectionError, ResponseError, TimeoutError, RedisServerError)),
    )
    def _check_ping_key(self, timestamp):
        val = self._cache.get(self._ping_key)
        if not val:
            raise RedisServerError('Cache fail, empty key "{!r}"'.format(self._ping_key))

        cached_timestamp = int(val)
        diff = timestamp - cached_timestamp
        if diff > self._maximum_lag:
            raise RedisServerError(
                'Cache fail, timestamp difference {!r}, maximum lag {!r} for "{!r}"'.format(
                    diff,
                    self._maximum_lag,
                    self._ping_key,
                )
            )

        return True


class RedisPing(object):
    def __init__(self, cache, ttl, sleep_time):
        self.cache = cache
        self.ttl = ttl
        self.sleep_time = sleep_time
        self.thr = threading.Thread(target=self._ping)
        self.alive = False

    def run(self):
        self.alive = True
        if not self.thr.is_alive():
            self.thr.start()

    def stop(self):
        self.alive = False
        if self.thr.is_alive():
            self.thr.join()

    def _ping(self):
        while self.alive:
            try:
                self._set_ping_key()
            except Exception:
                log.exception('Ping set key error')
            time.sleep(self.sleep_time / 1000)

    @retry(
        stop_max_attempt_number=3,
        wait_fixed=200,
        retry_on_exception=lambda e: isinstance(e, (ConnectionError, ResponseError, TimeoutError)),
    )
    def _set_ping_key(self):
        timestamp = int(time.time() * 1000.0)
        self.cache.set(RedisChecker.KEY_PING_CHECK, timestamp, self.ttl)


class SharedCacheChecker(object):
    def __init__(self, service_name, connection):
        self._service_name = service_name
        self._shared_cache = connection

    def check(self):
        # Доступность кэша
        random_key = '{}/ping_check_{}'.format(
            self._service_name, random.random(),
        )
        random_val = str(random.random())
        self._shared_cache.set(random_key, random_val, 30)
        restored_val = self._shared_cache.get(random_key)
        if restored_val != random_val:
            raise SharedCacheServerError(
                'Shared cache fail. Saved: {!r}. Loaded: {!r}'.format(
                    random_val,
                    restored_val,
                )
            )

        return 'alive'


class RedisChecker(object):
    KEY_PING_CHECK = '_ping_check_{}'.format(random.random())

    def __init__(self, redis_cache, maximum_lag):
        self._redis_cache = redis_cache
        self.maximum_lag = maximum_lag

    @retry(
        stop_max_attempt_number=3,
        wait_fixed=500,
        retry_on_exception=lambda e: isinstance(e, RedisServerError),
    )
    def check(self):
        val = self._redis_cache.get(self.KEY_PING_CHECK)
        if not val:
            raise RedisServerError('Cache fail, empty key "{!r}"'.format(self.KEY_PING_CHECK))

        cached_timestamp = int(val)
        timestamp = int(time.time() * 1000.0)
        diff = timestamp - cached_timestamp
        if diff >= self.maximum_lag:
            raise RedisServerError(
                'Cache fail, timestamp difference {!r}, maximum lag {!r} including sleep time'.format(
                    diff,
                    self.maximum_lag,
                )
            )

        return 'alive'
