# -*- coding: utf-8 -*-
from __future__ import unicode_literals
from typing import Optional  # noqa

import yt.wrapper as yt

from travel.avia.library.python.references.helpers import point_key_to_id


class StationCache(object):
    def __init__(self, rows):
        self._by_code = {}
        self._by_id = {}
        self._fill_maps(rows)

    def _fill_maps(self, rows):
        for station in rows:
            if station['city_id']:
                if station['iata']:
                    self._by_code[station['iata'].lower()] = station
                if station['sirena']:
                    self._by_code[station['sirena'].lower()] = station
            if station.get('id'):
                self._by_id[point_key_to_id(station['id'])] = station

    def settlement_id_by_code(self, code, raise_on_unknown=True):
        try:
            return self._by_code[code.lower()]['city_id']
        except KeyError:
            if raise_on_unknown:
                raise KeyError(
                    'Settlement for code {!r} is not found'.format(code)
                )
            return None

    def station_code_by_id(self, station_id, raise_on_unknown=True):
        try:
            station = self._by_id[station_id]
            return station.get('iata') or station.get('sirena')
        except KeyError:
            if raise_on_unknown:
                raise KeyError(
                    'Settlement for id {!r} is not found'.format(id)
                )
            return None

    def settlement_id_by_id(self, station_id, raise_on_unknown=True):
        # type: (int, bool) -> Optional[int]
        try:
            settlement_key = self._by_id[station_id]['city_id']
            if settlement_key:
                return point_key_to_id(settlement_key)
        except KeyError:
            if raise_on_unknown:
                raise KeyError(
                    'Settlement for id {!r} is not found'.format(station_id)
                )
            return None

    def station_timezone_by_station_code(self, code):
        try:
            return self._by_code[code.lower()]['time_zone']
        except KeyError:
            raise KeyError(
                'Settlement for code {!r} is not found'.format(code)
            )


def create_station_cache(yt_client, reference_yt_path='//home/rasp/reference'):
    # type: (yt.YtClient, str) -> StationCache
    rows = yt_client.read_table(yt.TablePath(yt.ypath_join(reference_yt_path, 'station')))
    return StationCache(rows)
