# -*- coding: utf-8 -*-

import re

from django.conf import settings
from django.db.models import Q

from travel.avia.library.python.common.models.schedule import DeLuxeTrain, Route
from travel.avia.library.python.common.models.transport import TransportType
from travel.avia.library.python.common.utils.caching import cached
from travel.avia.library.python.common.utils.text import transliterate, md5_hex
from travel.avia.library.python.route_search.by_number.models import RouteNumberIndex, ThreadSearchResult


def get_search_result(query, t_type_ids=None, search_for_special_train=True):
    t_type_ids = build_t_type_ids_list(t_type_ids)

    # если передан пустой номер, возвращаем пустой результат
    if not query:
        return ThreadSearchResult()

    if query.isnumeric():
        routes = _find_numeric_routes(query, t_type_ids)
        return ThreadSearchResult(routes)

    if search_for_special_train and get_t_type_id('train') in t_type_ids:
        special_train_routes = _find_special_train_routes(query)

        if special_train_routes:
            return ThreadSearchResult(special_train_routes, is_special_train=True)

    routes = _find_routes_by_wide_search(query, t_type_ids)

    return ThreadSearchResult(routes)


def _find_numeric_routes(number, t_type_ids):
    plane_number = number.lstrip(u'0')

    train_number = number.lstrip('0')
    train_number = '0' * (3 - len(train_number)) + train_number

    condition = (
        Q(rthread__t_type=TransportType.PLANE_ID, rthread__reversed_number__startswith=plane_number[::-1] + u' ') |
        Q(rthread__t_type=TransportType.BUS_ID, rthread__number=number)
    )

    if len(number) >= 3:
        condition |= (
            Q(rthread__t_type__in=[TransportType.BUS_ID, TransportType.SUBURBAN_ID, TransportType.HELICOPTER_ID] +
                                  TransportType.WATER_TTYPE_IDS,
              rthread__reversed_number__startswith=number[::-1]) |
            Q(rthread__t_type=TransportType.SUBURBAN_ID, rthread__number__startswith=number)
        )

    return list(
        (
            Route.objects.filter(rthread__t_type=TransportType.TRAIN_ID)
                         .extra(where=['`www_rthread`.`route_number` LIKE %s'], params=[train_number + '_']) |
            Route.objects.filter(condition)
        ).exclude(hidden=True).filter(rthread__t_type__in=t_type_ids)
    )


def _find_special_train_routes(query):
    special_train_query_exact = _get_special_train_query_exact(query)

    special_trains = DeLuxeTrain.objects.filter(special_train_query_exact)

    if len(query) > 3:
        special_train_query_startswith = _get_special_train_query_startwith(query)

        special_trains = special_trains or DeLuxeTrain.objects.filter(special_train_query_startswith)

    if not special_trains:
        return []

    numbers = set()

    for d_train in special_trains:
        numbers.update(d_train.numbers.split(u'/'))

    route_query = Q()

    for number in numbers:
        route_query |= Q(rthread__number=number)

    return list(Route.objects.filter(route_query).exclude(hidden=True))


def _get_special_train_query_exact(query):
    q = Q(title=query)

    for lang in settings.MODEL_LANGUAGES:
        q |= Q(**{'title_{}'.format(lang): query})

    return q


def _get_special_train_query_startwith(query):
    q = Q(title__startswith=query)

    for lang in settings.MODEL_LANGUAGES:
        q |= Q(**{'title_{}__startswith'.format(lang): query})

    return q


def _find_routes_by_wide_search(query, t_type_ids):
    # Варианты поисковых запросов с учетом транслита
    transliterated_number = transliterate(query, 'cyr-lat')
    trimmed_number = re.sub(ur'\W', u'', query, flags=re.U + re.I).strip().lower()
    zero_trimmed_number = _strange_trim_zeroes(trimmed_number)
    transliterated_trimmed_number = transliterate(trimmed_number, 'cyr-lat')
    zero_start_trimmed_number = query.lstrip(u'0')

    q = Q(number__istartswith=query)

    for var in [transliterated_number,
                trimmed_number,
                zero_trimmed_number,
                transliterated_trimmed_number,
                zero_start_trimmed_number]:

        if var:
            q = q | Q(number__istartswith=var)

    route_index = RouteNumberIndex.objects.filter(q) \
                                  .exclude(route__hidden=True)\
                                  .exclude(route__rthread__t_type=TransportType.BUS_ID)\
                                  .filter(route__rthread__t_type__in=t_type_ids)\
                                  .select_related('route')[:1000]

    return [r.route for r in route_index]


def _strange_trim_zeroes(number):
    # FIXME: для S7 не сканает
    return re.sub('(^|\D)0+(\d)', lambda res: res.group(1) + res.group(2), number)


@cached(lambda number: "get_threads/%s" % md5_hex(number),
        timeout=settings.CACHES['default']['TIMEOUT'])
def old_sub_find_threads(number):
    """ Подбирает список нитей по номеру рейса """

    number = number.strip()

    return get_search_result(number, search_for_special_train=False).threads


def build_t_type_ids_list(t_type_ids):
    # Если пустой список или None, то нужно искать по всем типам транспорта
    if not t_type_ids:
        t_type_ids = [t.id for t in TransportType.objects.all()]

    return t_type_ids


def get_t_type_id(code):
    try:
        return TransportType.objects.get(code=code).id

    except (TransportType.DoesNotExist, TransportType.MultipleObjectsReturned):
        return None


def get_t_type_ids(codes):
    return [t.id for t in TransportType.objects.filter(code__in=codes)]


def number_variants(number):
    pattern = re.compile('\W', re.UNICODE)
    trimmed_number = re.sub(pattern, '', number)

    triv = lambda x: x
    dash = lambda x: x.replace(' ', '-')
    trim_spaces = lambda x: re.sub('\s', '', x)
    zero = lambda x: re.sub('\s(\d)', lambda res: u' 0%s' % res.group(1), x)
    zero2 = lambda x: re.sub('\s(\d)', lambda res: u' 00%s' % res.group(1), x)

    variants = []
    for num in number, trimmed_number:
        for f in (triv, dash, trim_spaces):
            for g in (triv, zero, zero2):
                variants.append(f(g(num)))
                variants.append(f(g(_strange_trim_zeroes(num))))
                variants.append(f(g(transliterate(num, 'cyr-lat'))))
                variants.append(f(g(transliterate(num, 'lat-cyr'))))

    return set(variants)
