# -*- coding: utf-8 -*-

from itertools import groupby, chain

from django.db.models import Q

from travel.avia.library.python.common.models.geo import StationMajority, Settlement, Station
from travel.avia.library.python.common.models.transport import TransportType


TRAIN_T_TYPE_ID = 1
BASIC_THREAD_TYPE_ID = 1


def remove_duplicates(threads):
    """
    Группируем по id реального треда (наши треды не совсем настоящие, так они получаются спецвыборкой)
    :param threads:
    :return:
    """
    threads.sort(key=lambda t: t.id)

    for base_id, threads in groupby(threads, key=lambda t: t.id):
        yield best_thread(threads)


def best_thread(duplicates):
    """
    Получаем первым элементом рейс от станции с максимальным приоритеом до станции с максимальным приоритетом,
    если станций с максимальным приоритетом несколько, то от самой поздней до самой ранней из них.
    """

    return min(duplicates, key=lambda t: (
        t.station_from.majority_id,  # Минимальное числовое значение приоритета
        t.station_to.majority_id,  # Минимально числовое значение приоритета
        t.rtstation_from.tz_arrival is not None,  # Начальная станция маршрута
        -(t.rtstation_from.tz_departure or 0),  # Из равнозначных станции в городе, берем с
                                                # самым поздним отправлением
        t.rtstation_to.tz_arrival,  # Минимальное значение времени
    ))


def remove_through_trains(presegments):
    """
    Удаляем безпересадочные нитки, у которых в поиск попала основная нитка.
    Предполагаем, что нитки импортированы с одинаковыми временными зонами.
    Иначе они попадут в разные группы threads_by_rts_time_zone, и мы не удалим дубликат.
    """

    def key_func(prs):
        t = prs.thread
        return (
            t.t_type_id,
            t.station_from_id, prs.rts_from_departure_time,
            t.station_to_id, prs.pseudo_duration_td,
        )

    presegments.sort(key=key_func)

    for key, presegments in groupby(presegments, key=key_func):
        t_type_id = key[0]

        presegments = list(presegments)

        if t_type_id == TRAIN_T_TYPE_ID and len(presegments) > 1:
            basic = [p for p in presegments if p.thread.type_id == BASIC_THREAD_TYPE_ID]

            if basic:
                presegments = basic

        for presegm in presegments:
            yield presegm


class LimitConditions(object):
    def __init__(self, point_from, point_to, t_types=None, extended=False):
        """
        Если первая и вторая точка, это станции из одного города,
        или одна из них город, а другая станция этого города,
        исключаем поезда дальнего следования (ПДС) и самолеты из выдачи.
        Иначе ищем по всему.

        Из поиска исключаем станции с важностью меньше чем
        не попадает в табло, если ищем между городами.
        """

        t_types = t_types or TransportType.objects.all_cached()

        assert point_from != point_to
        self.point_from = point_from
        self.point_to = point_to

        new_t_types = None

        train_plane = (TransportType.objects.get(code='train'), TransportType.objects.get(code='plane'))
        max_majority_id = StationMajority.objects.get(code='station').id

        from_max_majority_id = to_max_majority_id = max_majority_id

        is_different_settlements = False
        is_stations_in_the_same_city = False
        is_from_city_to_station_in_other_city = False
        is_from_city_to_station_in_the_same_city = False
        is_from_station_in_other_city_to_city = False
        is_from_station_in_the_same_city_to_city = False

        if isinstance(point_from, Settlement) and isinstance(point_to, Settlement):
            is_different_settlements = True

        elif isinstance(point_from, Station) and isinstance(point_to, Station):
            if point_from.settlement_id == point_to.settlement_id and \
               point_from.settlement_id is not None:
                is_stations_in_the_same_city = True

        elif isinstance(point_from, Station):
            if point_from.settlement_id == point_to.id:
                is_from_station_in_the_same_city_to_city = True

            else:
                is_from_station_in_other_city_to_city = True

        elif isinstance(point_to, Station):
            if point_to.settlement_id == point_from.id:
                is_from_city_to_station_in_the_same_city = True

            else:
                is_from_city_to_station_in_other_city = True

        if is_different_settlements:
            from_max_majority_id = to_max_majority_id = StationMajority.objects.get(code='not_in_tablo').id

        # Если поиск в рамках города ограничиваем типы транспорта
        elif (is_stations_in_the_same_city
                or is_from_city_to_station_in_the_same_city
                or is_from_station_in_the_same_city_to_city):

            if not extended:
                new_t_types = [t_type for t_type in t_types if t_type not in train_plane]

        elif is_from_city_to_station_in_other_city:
            from_max_majority_id = StationMajority.objects.get(code='not_in_tablo').id

        elif is_from_station_in_other_city_to_city:
            to_max_majority_id = StationMajority.objects.get(code='not_in_tablo').id

        if new_t_types:
            t_types = new_t_types

        self.from_max_majority_id = from_max_majority_id
        self.to_max_majority_id = to_max_majority_id

        self.t_types = t_types

    def allow(self, from_majority_id, to_majority_id, t_type):
        if t_type not in self.t_types:
            return False

        if t_type.code == 'bus':
            return True

        if from_majority_id > self.from_max_majority_id:
            return False

        if to_majority_id > self.to_max_majority_id:
            return False

        return True

    def filter_threads_qs(self, threads_qs):
        point_from, point_to = self.point_from, self.point_to

        if (
            isinstance(point_from, Station) and point_from.majority_id > self.from_max_majority_id
        ) or (
            isinstance(point_to, Station) and point_to.majority_id > self.to_max_majority_id
        ):
            return threads_qs.none()

        query = Q(t_type__in=self.t_types)

        if isinstance(point_from, Settlement):
            query &= Q(znoderoute2__station_from__in=list(chain(
                point_from.station_set.filter(majority__lte=self.from_max_majority_id).values_list('id', flat=True),
                point_from.station2settlement_set.filter(station__majority__lte=self.from_max_majority_id)
                                                 .values_list('station_id', flat=True)
            )))
        else:
            query &= Q(znoderoute2__station_from=point_from)

        if isinstance(point_to, Settlement):
            query &= Q(znoderoute2__station_to__in=list(chain(
                point_to.station_set.filter(majority__lte=self.to_max_majority_id).values_list('id', flat=True),
                point_to.station2settlement_set.filter(station__majority__lte=self.to_max_majority_id)
                                               .values_list('station_id', flat=True)
            )))
        else:
            query &= Q(znoderoute2__station_to=point_to)

        return threads_qs.filter(query)


def fetch_unrelated(objects, unrelated_model, *fields):
    """Фетчим модели полей, которых в изначальной модели нет"""

    unrelated = set(getattr(o, field + '_id') for o in objects for field in fields)

    unrelated_objects = unrelated_model._default_manager.in_bulk(list(unrelated))

    for o in objects:
        for field in fields:
            setattr(o, field, unrelated_objects.get(getattr(o, field + '_id')))


def _t_type2t_type_list(t_type):
    if t_type is None:
        return [t for t in TransportType.objects.all_cached()]

    if isinstance(t_type, basestring):
        return [TransportType.objects.get(code=t_type)]

    if isinstance(t_type, TransportType):
        return [t_type]

    if isinstance(t_type[0], basestring):
        return [TransportType.objects.get(code=code) for code in t_type]

    if isinstance(t_type[0], TransportType):
        return t_type
