# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging
from copy import copy, deepcopy
from datetime import datetime

from django.db import models
from django.utils.functional import cached_property
from django.utils.translation import ugettext_lazy as _

from travel.avia.library.python.common.models_utils import DisplayTCodeMixin
from travel.avia.library.python.common.models_utils.i18n import RouteLTitle, L_stops_mixin
from travel.avia.library.python.common.utils import environment
from travel.avia.library.python.common.utils.date import FuzzyDateTime, MSK_TZ
from travel.avia.library.python.common.utils.date import smart_localize
from travel.avia.library.python.common.utils.models import SwapModel
from travel.avia.library.python.route_search.models_abstract import BaseZNodeRoute2


log = logging.getLogger(__name__)


class ZNodeRoute2(BaseZNodeRoute2):
    __metaclass__ = SwapModel
    admin_module_path = 'travel.avia.admin.www_admin.models.route_search'

    class Meta(BaseZNodeRoute2.Meta):
        app_label = 'www'
        db_table = 'www_znoderoute2'


class BaseThreadSegment(L_stops_mixin, DisplayTCodeMixin):
    thread = None
    station_from = None
    station_to = None

    @cached_property
    def t_type(self):
        return self.thread.t_type

    @cached_property
    def company(self):
        return self.thread.company

    def L_title(self, **kwargs):
        return self.thread.L_title(**kwargs)

    def get_popular_title(self, lang=None):
        return self.thread.get_popular_title(lang=lang)

    @cached_property
    def t_model(self):
        return self.thread.t_model

    def path(self):
        """Проходит по пути, возвращая только станции, принадлежащие сегменту"""

        path = iter(self.thread.path)

        for rts in path:
            if rts.station_id != self.station_from.id:
                continue

            yield rts
            break

        for rts in path:
            yield rts

            if rts.station_id == self.station_to.id:
                break

    @classmethod
    def fetch_titles(cls, segments):
        threads = []

        for segment in segments:
            thread = getattr(segment, 'thread', None)

            if thread:
                threads.append(thread)

            if hasattr(segment, 'segments'):
                for subsegment in segment.segments:
                    thread = getattr(subsegment, 'thread', None)
                    if thread:
                        threads.append(thread)

        RouteLTitle.fetch([t.L_title for t in threads])

    def days_text(self, *args, **kwargs):
        # Расписание по отображаемому времени
        if 'extra_link' in kwargs:
            kwargs['extra_link'] = kwargs['extra_link'] % self.run_days_link()

        return self.thread.days_text(self.mask_shift, *args, **kwargs)

    @cached_property
    def mask_shift(self):
        return self.rtstation_from.calc_days_shift(
            event='departure',
            start_date=self.start_date,
            event_tz=self.rtstation_from.station.pytz
        )

    @cached_property
    def calculated_start_date(self):
        """
        Вычисляем дату старта, если ее не заполнили
        """
        if self.start_date:
            return self.start_date

        departure_tz = self.rtstation_from.station.pytz
        departure_date = self.departure.astimezone(departure_tz).date()

        start_date = self.rtstation_from.calc_thread_start_date(
            event='departure',
            event_date=departure_date,
            event_tz=departure_tz
        )

        return start_date

    @cached_property
    def start_dt(self):
        assert self.thread is not None

        naive_start_dt = datetime.combine(self.calculated_start_date, self.thread.tz_start_time)
        return smart_localize(naive_start_dt, self.thread.pytz)

    @property
    def is_express(self):
        return self.thread.is_express

    @cached_property
    def duration(self):
        return self.arrival - self.departure


class RThreadSegment(BaseThreadSegment):
    def __init__(self):
        self.info = None
        self.station_from = None
        self.station_to = None
        self.rtstation_from = None
        self.rtstation_to = None
        self.departure = None
        self.arrival = None
        self.thread = None
        self.start_date = None
        self.supplier_code = None
        self.now = None

        self.is_fuzzy_from = False
        self.is_fuzzy_to = False

        self.departure_z_tablo = None
        self.arrival_z_tablo = None

    def _init_data(self):
        # Аттрибуты работы из шаблонов
        # Для пересадочного thread_segment
        if self.thread is not None:
            self.title = self.thread.title
            self.number = self.thread.number
            self.t_type = self.thread.t_type
            self.company = self.thread.company

            if self.thread.supplier:
                self.supplier_code = self.thread.supplier.code

        if self.station_from and self.station_to:
            if self.now:
                self.gone = self.departure < self.now

        self.is_fuzzy_from = self.rtstation_from and self.rtstation_from.is_fuzzy

        if self.is_fuzzy_from:
            if self.departure:
                self.departure = FuzzyDateTime.fuzz(self.departure)

        self.is_fuzzy_to = self.rtstation_to and self.rtstation_to.is_fuzzy

        if self.is_fuzzy_to:
            if self.arrival:
                self.arrival = FuzzyDateTime.fuzz(self.arrival)

    @cached_property
    def msk_departure(self):
        return self.departure.astimezone(MSK_TZ)

    @cached_property
    def msk_arrival(self):
        return self.arrival.astimezone(MSK_TZ)

    def is_valid(self):
        return (
            self.thread
            and self.arrival is not None
            and self.departure is not None
            and self.station_from is not None
            and self.station_to is not None
        )

    def copy(self):
        copy_of_self = copy(self)

        copy_of_self.info = deepcopy(self.info)

        return copy_of_self


class AllDaysRThreadSegment(RThreadSegment):
    def _init_data(self):
        super(AllDaysRThreadSegment, self)._init_data()

        self.gone = False


class IntervalRThreadSegment(RThreadSegment):
    last_departure = None
    msk_last_departure = None

    def _init_data(self):
        """
        Для интервальной нитки не заполняем отправление и прибытие
        """
        if self.thread is not None:
            self.title = self.thread.title
            self.number = self.thread.number
            self.t_type = self.thread.t_type
            self.company = self.thread.company

            if self.thread.supplier:
                self.supplier_code = self.thread.supplier.code

            self.gone = (self.last_departure is not None and
                         self.last_departure < environment.now_aware())


class RThreadSegmentGroup(object):
    def __init__(self, number, title, departure_time, arrival_time, mask, segments):
        self.number = number
        self.title = title
        self.departure_time = departure_time
        self.arrival_time = arrival_time
        self.mask = mask

        if not segments:
            raise ValueError('RThreadSegments should have at least one segment')

        self.segments = segments

    def __iter__(self):
        return iter(self.segments)


class NearestSuburbanDirection(models.Model):
    settlement = models.OneToOneField('www.Settlement', verbose_name=_('Город пользователя'), on_delete=models.CASCADE,
                                      null=False)
    station = models.ForeignKey('www.Station', related_name='+',
                                on_delete=models.CASCADE,
                                verbose_name=_('Станция города пользователя'), null=False)
    transport_center = models.ForeignKey('www.Station', related_name='+',
                                         on_delete=models.CASCADE, null=False,
                                         verbose_name=_('Станция транспортного центра'),
                                         help_text=_('Станция транспортного центра до которой, есть электричечное'
                                                     ' сообщение из города пользователя'))

    class Meta:
        app_label = 'route_search'
        verbose_name = _('Направление ближайших электричек')
        verbose_name_plural = _('Направления ближайших электричек')
