# coding=utf-8
import logging
import json
from typing import Type, Optional

from requests import Session
from requests.adapters import HTTPAdapter
from urllib3 import Retry

from sandbox import sdk2
from sandbox.common import rest
from sandbox.common import proxy

logger = logging.getLogger(__name__)


class ResourceFetcher(object):
    def __init__(self, sandbox_client, sandbox_oauth_token=None):
        # type: (rest.Client, proxy.OAuth) -> None
        self._client = sandbox_client
        self._oauth_token = sandbox_oauth_token.token if sandbox_oauth_token else None
        self._session = Session()
        self._session.mount('https://', HTTPAdapter(max_retries=Retry(
            total=3,
            read=3,
            connect=3,
            backoff_factor=0.1,
            status_forcelist=(500, 502, 503, 504),
        )))

    def fetch_latest_ready(self, resource_class):
        # type: (Type[sdk2.Resource]) -> Optional[str]
        raw_resource = self._client.resource.read(type=resource_class.__name__, state='READY', order='-id', limit=1)

        if not raw_resource or not raw_resource.get('items') or not raw_resource['items'][0]:
            logger.warning(
                'Unable to fetch the transport models data. Result: %s.',
                json.dumps(raw_resource)
            )
            return None

        resource_info = raw_resource['items'][0]
        resource_http = resource_info.get('http')
        resource_proxy = resource_http.get('proxy') if resource_http else None
        if not resource_proxy:
            logger.info(
                'Unable to get http proxy URL for the transport models data. Result: %s.',
                json.dumps(resource_info)
            )
            return None

        logger.info('loading resource data from %s', resource_proxy)
        headers = {}
        if self._oauth_token:
            headers['Authorization'] = 'OAuth {}'.format(self._oauth_token)
        return self._session.get(resource_proxy, headers=headers).content
