# encoding: utf-8
from collections import namedtuple

from travel.avia.library.python.shared_dicts.rasp import iter_protobuf_data, ResourceType

CacheBuilderResult = namedtuple('CacheBuilderResult', ['country_by_id', 'country_by_code3', 'country_by_code'])


class CountryCache:
    SANDBOX_RESOURCE = ResourceType.TRAVEL_DICT_RASP_COUNTRY_PROD

    def __init__(self, logger):
        self.country_by_id = {}
        self.country_by_code3 = {}
        self.country_by_code = {}
        self.logger = logger

    def get_country_by_id(self, country_id):
        if not country_id:
            return None
        if country_id in self.country_by_id:
            return self.country_by_id[country_id]
        self.logger.warning('No country for country_id %d in the cache', country_id)

    def get_country_by_code3(self, code3):
        if not code3:
            return None
        return self.country_by_code3.get(code3.upper())

    def get_country_by_code(self, code):
        if not code:
            return None
        return self.country_by_code.get(code.upper())

    def clear_cache(self):
        self.country_by_id = {}
        self.country_by_code3 = {}
        self.country_by_code = {}

    def populate(self):
        cache = self.build_cache()
        self.country_by_id = cache.country_by_id
        self.country_by_code3 = cache.country_by_code3
        self.country_by_code = cache.country_by_code
        self.logger.info(
            'repopulated cache with %d countries, %d 3-letter codes, %d 2-letter codes',
            len(self.country_by_id),
            len(self.country_by_code3),
            len(self.country_by_code),
        )

    def update_cache(self):
        try:
            self.populate()
        except Exception as e:
            self.logger.error('Failed to update country cache. Will continue using old cache. Reason: %s', e)

    @classmethod
    def build_cache(cls):
        country_by_id = {}
        country_by_code3 = {}
        country_by_code = {}
        for country in iter_protobuf_data(CountryCache.SANDBOX_RESOURCE):
            country_by_id[country.Id] = country
            if country.Code3:
                country_by_code3[country.Code3.upper()] = country
            if country.Code:
                country_by_code[country.Code.upper()] = country
        return CacheBuilderResult(
            country_by_id=country_by_id,
            country_by_code3=country_by_code3,
            country_by_code=country_by_code,
        )
