# encoding: utf-8
from six import ensure_text

from travel.avia.library.python.shared_dicts.rasp import iter_protobuf_data, ResourceType
from travel.avia.library.python.slug.slug import build_slug


class SettlementCache:
    SANDBOX_RESOURCE = ResourceType.TRAVEL_DICT_RASP_SETTLEMENT_PROD

    def __init__(self, logger):
        self.settlement_by_id = {}
        self.settlement_by_geo_id = {}
        self.settlement_by_slug = {}
        self.slug_by_id = {}
        self.settlement_by_iata = {}
        self.settlement_by_sirena_id = {}
        self.logger = logger

    def get_settlement_by_id(self, settlement_id):
        if not settlement_id:
            return None
        if settlement_id in self.settlement_by_id:
            return self.settlement_by_id[settlement_id]
        self.logger.warning('No settlement for settlement_id %d in the cache', settlement_id)

    def get_settlement_by_geo_id(self, geo_id):
        if not geo_id:
            return None
        if geo_id in self.settlement_by_geo_id:
            return self.settlement_by_geo_id[geo_id]
        self.logger.warning('No settlement for geo_id %s in the cache', geo_id)

    def get_settlement_by_slug(self, slug):
        if not slug:
            return None
        if slug in self.settlement_by_slug:
            return self.settlement_by_slug[slug]
        self.logger.warning('No settlement for slug %s in the cache', slug)

    def get_settlement_by_iata(self, iata):
        if not iata:
            return None
        if iata in self.settlement_by_iata:
            return self.settlement_by_iata[iata]
        self.logger.warning('No settlement for iata %s in the cache', iata)

    def get_settlement_by_sirena_id(self, sirena_id):
        if not sirena_id:
            return None
        if sirena_id in self.settlement_by_sirena_id:
            return self.settlement_by_sirena_id[sirena_id]
        self.logger.warning('No settlement for sirena_id %s in the cache', sirena_id)

    def get_slug_by_id(self, settlement_id):
        if not settlement_id:
            return None
        if settlement_id in self.slug_by_id:
            return self.slug_by_id[settlement_id]
        self.logger.warning('No slug for settlement_id %d in the cache', settlement_id)

    def clear_cache(self):
        self.settlement_by_id = {}
        self.settlement_by_slug = {}
        self.slug_by_id = {}

    def populate(self):
        (
            self.settlement_by_id,
            self.settlement_by_geo_id,
            self.settlement_by_slug,
            self.slug_by_id,
            self.settlement_by_iata,
            self.settlement_by_sirena_id,
        ) = self.build_cache()
        self.logger.info(
            'populated cache with %d settlements and %d slugs',
            len(self.settlement_by_id),
            len(self.settlement_by_slug),
        )

    def update_cache(self):
        try:
            (
                self.settlement_by_id,
                self.settlement_by_geo_id,
                self.settlement_by_slug,
                self.slug_by_id,
                self.settlement_by_iata,
                self.settlement_by_sirena_id,
            ) = self.build_cache()
        except Exception as e:
            self.logger.error('Failed to update settlement cache. Will continue using old cache. Reason: %s', e)

        self.logger.info(
            'repopulated cache with %d settlements and %d slugs',
            len(self.settlement_by_id),
            len(self.settlement_by_slug),
        )

    @classmethod
    def build_cache(cls):
        settlement_by_id = {}
        settlement_by_geo_id = {}
        slug_by_id = {}
        settlement_by_slug = {}
        settlement_by_iata = {}
        settlement_by_sirena_id = {}

        for settlement in iter_protobuf_data(SettlementCache.SANDBOX_RESOURCE):
            settlement_by_id[settlement.Id] = settlement
            settlement_by_geo_id[settlement.GeoId] = settlement
            slug = cls._get_slug(settlement)
            settlement_by_slug[slug] = settlement
            slug_by_id[settlement.Id] = slug

            if settlement.Iata:
                settlement_by_iata[settlement.Iata] = settlement
            if settlement.SirenaId:
                settlement_by_sirena_id[ensure_text(settlement.SirenaId)] = settlement

        return (
            settlement_by_id,
            settlement_by_geo_id,
            settlement_by_slug,
            slug_by_id,
            settlement_by_iata,
            settlement_by_sirena_id,
        )

    @staticmethod
    def _get_slug(settlement):
        if settlement.Slug:
            return settlement.Slug
        if settlement.Title.En.Nominative:
            title = settlement.Title.En.Nominative
        else:
            title = settlement.TitleDefault
        title_utf8 = ensure_text(title)
        return build_slug(title_utf8)
