from travel.avia.library.python.shared_dicts.rasp import iter_protobuf_data, ResourceType


class StationCodeCache:
    SANDBOX_RESOURCE = ResourceType.TRAVEL_DICT_RASP_STATION_CODE_PROD
    # codes from enum in protobuf
    CODE_SYSTEM_SIRENA = 1
    CODE_SYSTEM_IATA = 4
    CODE_SYSTEM_ICAO = 5

    def __init__(self, logger):
        self.station_id_to_iata = {}
        self.station_id_to_icao = {}
        self.station_id_to_sirena = {}
        self.logger = logger

    def get_station_code_by_id(self, station_id):
        if not station_id:
            return None
        if station_id in self.station_id_to_iata:
            return self.station_id_to_iata[station_id]
        if station_id in self.station_id_to_icao:
            return self.station_id_to_icao[station_id]
        if station_id in self.station_id_to_sirena:
            return self.station_id_to_sirena[station_id]
        self.logger.warning('No code for station_id %d in the cache', station_id)

    def get_station_id_by_code(self, station_code):
        if station_code in self.iata_to_station_id:
            return self.iata_to_station_id[station_code]
        if station_code in self.sirena_to_station_id:
            return self.sirena_to_station_id[station_code]
        self.logger.warning('No station id for station_code %s in the cache', station_code)

    def clear_cache(self):
        self.station_id_to_iata = {}
        self.station_id_to_icao = {}
        self.station_id_to_sirena = {}
        self.iata_to_station_id = {}
        self.sirena_to_station_id = {}

    def populate(self):
        (
            self.station_id_to_iata,
            self.station_id_to_icao,
            self.station_id_to_sirena,
            self.iata_to_station_id,
            self.sirena_to_station_id,
        ) = self.build_cache()
        self.logger.info(
            'populated cache with %d iatas, %d icaos, %d sirena_ids',
            len(self.station_id_to_iata), len(self.station_id_to_icao), len(self.station_id_to_sirena)
        )

    def update_cache(self):
        try:
            (
                self.station_id_to_iata,
                self.station_id_to_icao,
                self.station_id_to_sirena,
                self.iata_to_station_id,
                self.sirena_to_station_id,
            ) = self.build_cache()
        except Exception as e:
            self.logger.error('Failed to update station codes cache. Will continue using old cache. Reason: %s', e)

        self.logger.info(
            'repopulated cache with %d iatas, %d icaos, %d sirena_ids',
            len(self.station_id_to_iata), len(self.station_id_to_icao), len(self.station_id_to_sirena)
        )

    @staticmethod
    def build_cache():
        station_id_to_iata = {}
        iata_to_station_id = {}
        station_id_to_icao = {}
        station_id_to_sirena = {}
        sirena_to_station_id = {}

        station_codes = iter_protobuf_data(StationCodeCache.SANDBOX_RESOURCE)
        for station_code in station_codes:
            if station_code.SystemId == StationCodeCache.CODE_SYSTEM_IATA:
                station_id_to_iata[station_code.StationId] = station_code.Code
                iata_to_station_id[station_code.Code] = station_code.StationId
            elif station_code.SystemId == StationCodeCache.CODE_SYSTEM_ICAO:
                station_id_to_icao[station_code.StationId] = station_code.Code
            elif station_code.SystemId == StationCodeCache.CODE_SYSTEM_SIRENA:
                station_id_to_sirena[station_code.StationId] = station_code.Code
                sirena_to_station_id[station_code.Code] = station_code.StationId

        return station_id_to_iata, station_id_to_icao, station_id_to_sirena, iata_to_station_id, sirena_to_station_id
