import logging
import requests
import json

from sandbox.common import rest

from travel.library.python.dicts.base_repository import BaseRepository, BaseListRepository
from travel.avia.library.python.shared_dicts.exceptions import GetProtoResourceError

log = logging.getLogger(__name__)

PROTO_RESOURCE_GET_TIMEOUT = 15


def get_resource_proxy(resource_type_to_repo, resource_type):
    if resource_type not in resource_type_to_repo:
        raise ValueError('No repository registered for resource type {}'.format(resource_type))
    sandbox = rest.Client()
    resource = sandbox.resource.read(
        type=resource_type.value, state='READY', order='-id', limit=1)

    if not resource or not resource.get('items') or not resource['items'][0] or not resource['items'][0].get('id'):
        log.warning('Unable to fetch resource. Result: %s.', json.dumps(resource))
        return

    resource_http = resource['items'][0].get('http')
    resource_proxy = resource_http.get('proxy') if resource_http else None
    if not resource_proxy:
        log.warning(
            'Unable to get http proxy URL for %s. Result: %s.',
            resource_type,
            json.dumps(resource)
        )
        return

    return resource_proxy


def get_repository(resource_type_to_repo, resource_type, oauth=None):
    resource_proxy = get_resource_proxy(resource_type_to_repo, resource_type)
    if not resource_proxy:
        return

    repo = resource_type_to_repo[resource_type]()
    headers = {}
    if oauth:
        headers['Authorization'] = 'OAuth {}'.format(oauth)
    repo.load_from_string(_retrieve_and_check(resource_proxy, headers=headers))
    return repo


def get_binary_file(binary_file_url, oauth=None):
    headers = {}
    if oauth:
        headers['Authorization'] = 'OAuth {}'.format(oauth)
    return _retrieve_and_check(binary_file_url, headers=headers)


def iter_protobuf_data(resource_type_to_repo, resource_type, oauth=None):
    repo = get_repository(resource_type_to_repo, resource_type, oauth=oauth)
    if isinstance(repo, BaseRepository):
        for obj in repo.itervalues():
            yield obj
    elif isinstance(repo, BaseListRepository):
        for obj in repo.values():
            yield obj
    else:
        raise ValueError('Iteration over repos of type {} is not supported'.format(repo.__class__.__name__))


def _retrieve_and_check(url, headers):
    error_message = 'Error getting proto resource'
    try:
        response = requests.get(url, headers=headers, timeout=PROTO_RESOURCE_GET_TIMEOUT)
        response.raise_for_status()
    except Exception:
        log.exception(error_message)
        raise GetProtoResourceError(error_message)

    return response.content
