# -*- coding: utf-8 -*-
import logging
from collections import namedtuple
from datetime import date
from urllib3 import Retry

try:
    from urllib.parse import urljoin
except ImportError:
    from urlparse import urljoin

from requests import Response, Session
from requests.adapters import HTTPAdapter
from typing import Dict, Optional, Iterable, List

logger = logging.getLogger(__name__)

DATE_FORMAT = '%Y-%m-%d'
SHARED_FLIGHTS_API_URI = 'http://shared-flights.production.avia.yandex.net'

OperatingFlight = namedtuple('OperatingFlight', 'airlineID number title')


class SharedFlightsClient(object):
    def __init__(self, api_uri=SHARED_FLIGHTS_API_URI):
        self._api_uri = api_uri
        self._session = Session()
        self._session.mount(
            'http://',
            HTTPAdapter(
                max_retries=Retry(
                    total=3,
                    read=3,
                    connect=3,
                    backoff_factor=0.1,
                    status_forcelist=(500, 502, 503, 504),
                )
            ),
        )

    def flight_p2p_segment_info(self, from_codes, to_codes):
        raw_response = self._session.get(
            urljoin(self._api_uri, '/api/v1/flight-p2p-segment-info'),
            params={'from': from_codes, 'to': to_codes},
        )
        raw_response.raise_for_status()
        return raw_response.json()

    def flight_p2p_summary(self):
        raw_response = self._session.get(urljoin(self._api_uri, '/api/v1/flight-p2p-summary'))
        raw_response.raise_for_status()
        return raw_response.json()

    def get_operating_flight(self, company_code, number, departure_day):
        # type: (str, str, date)->Optional[OperatingFlight]
        # TODO: find cache for py2 and py3 or extract (and check py3 compat of) memoize from ticket daemon library
        flight = self.flight(
            company_code,
            number,
            departure_day,
        )
        if flight is None:
            return None
        if flight.get('operating'):
            flight = flight['operating']

        return OperatingFlight(
            airlineID=flight['airlineID'],  # ex.: 26 for SU
            number=flight['number'],  # "1404" (no carrier ID)
            title=flight['title'],  # "SU 1404" (displayable flight number)
        )

    def flight(self, company_code, number, departure_day):
        # type: (str, str, date) -> Optional[dict]
        raw_response = self._session.get(
            u'{base_url}/{iata}/{number}/{departure_day}/'.format(
                base_url=urljoin(self._api_uri, '/api/v1/flight'),
                iata=company_code,
                number=number,
                departure_day=departure_day.strftime('%Y-%m-%d'),
            ),
        )

        if raw_response.status_code == 404:
            return None

        raw_response.raise_for_status()

        return raw_response.json()

    def flights(self, flights):
        # type: (List[str]) -> List[dict]
        raw_response = self._session.post(urljoin(self._api_uri, '/api/v1/flights'), json={'flight': ','.join(flights)})

        raw_response.raise_for_status()

        return raw_response.json()

    def close_dates_with_direct_flight(self, forward_date, backward_date, airports_from, airports_to, timeout=0.5):
        # type: (date, date, list[int], list[int]) -> bool
        params = {
            'forward': forward_date.strftime(DATE_FORMAT),
            'from': airports_from,
            'to': airports_to,
        }
        if backward_date:
            params['backward'] = backward_date.strftime(DATE_FORMAT)
        raw_response = self._session.get(
            urljoin(self._api_uri, '/api/v1/flight-p2p-by-date'),
            params=params,
            timeout=timeout,
        )
        self.raise_for_status(raw_response)
        return raw_response.json()

    def flight_range_accessible(self, flights, national_version):
        # type: (Iterable[str], str) -> Dict[str, bool]
        raw_response = self._session.post(
            urljoin(self._api_uri, '/api/v1/flight-range-accessible'),
            json={
                'flight_numbers': list(flights),
                'national_version': national_version,
            },
        )
        raw_response.raise_for_status()

        data = raw_response.json()

        return {flight_number: data[flight_number]['Valid'] for flight_number in data}

    def flights_network(self, carriers):
        # type: (list[str]) -> dict
        raw_response = self._session.get(
            urljoin(self._api_uri, '/api/v1/flights-network'),
            params={'carrier': carriers},
        )
        self.raise_for_status(raw_response)
        return raw_response.json()

    def aeroflot_variants(self, depart_from, arrive_to, after, before, national_version, is_one_way):
        # type: (list[str], list[str], str, str, str, bool) -> dict
        params = {
            'from': depart_from,
            'to': arrive_to,
            'one_way': is_one_way,
        }
        if after:
            params['after'] = after
        if before:
            params['before'] = before
        if national_version:
            params['national_version'] = national_version
        raw_response = self._session.get(
            urljoin(self._api_uri, '/api/v1/aeroflot-variants'),
            params=params,
        )
        self.raise_for_status(raw_response)
        return raw_response.json()

    def raise_for_status(self, response):
        # type: (Response) -> None
        if response.status_code == 400:
            raise ClientBadRequestError(response.text or 'Invalid request')
        if response.status_code == 404:
            raise ClientNotFoundError(response.text or 'Not found some of the requested IDs')
        response.raise_for_status()


class ClientBadRequestError(Exception):
    pass


class ClientNotFoundError(Exception):
    pass
