# -*- coding: utf-8 -*-

import warnings
from datetime import date, datetime, timedelta

import pytz
from django.conf import settings
from django.utils.functional import cached_property
from django.utils.http import urlencode

from travel.avia.library.python.common.utils import environment
from travel.avia.library.python.common.utils.caching import cache_method_result
from travel.avia.library.python.common.utils.date import RunMask
from travel.avia.library.python.common.utils.locations import url
from travel.avia.library.python.common.utils.warnings import RaspDeprecationWarning
from travel.avia.library.python.stationschedule.utils import EVENT_ARRIVAL, EVENT_DEPARTURE, settlement_or_station_title


_fake_date = date.today()


class AbstractScheduleRoute(object):
    # Не связанные со временем аттрибуты

    z_tablo = None
    next_terminal = None

    @property
    def terminal(self):
        """
        @Deprecated
        Двусмысленное название, терминалы у нас только у аэропортов
        """
        warnings.warn("Use self.is_last_station, was remarked at 2015-02-28", RaspDeprecationWarning, stacklevel=2)

        return self.is_last_station

    @property
    def cancel(self):
        return self.thread.cancel

    @property
    def company(self):
        return self.thread.company

    def days_text(self, *args, **kwargs):
        return self.thread and self.thread.days_text(self.mask_shift, *args, **kwargs)

    @cached_property
    def displayed_number(self):
        return self.thread and self.thread.supplier.code in settings.SUPPLIERS_CODES_SHOW_NUMBER and self.number or None

    @cached_property
    def display_t_code(self):
        if self.thread:
            if self.thread.is_express:
                return 'express'

            if self.thread.is_aeroexpress:
                return 'aeroexpress'

            t_model = getattr(self.thread, 't_model', None)

            if t_model:
                return t_model.t_type.code

        return self.t_type.code

    @cached_property
    def extra_link(self):
        path = url('thread', self.thread.uid)

        params = {}

        if self.real_event == EVENT_DEPARTURE:
            params['station_from'] = self.station.id

        else:
            params['station_to'] = self.station.id

        return path + '?' + urlencode(params)

    def L_days_text(self, *args, **kwargs):
        kwargs['thread_start_date'] = self.naive_start_dt.date()

        return self.thread and self.thread.L_days_text(self.mask_shift, *args, **kwargs)

    def L_platform(self, lang=None):
        return self.rtstation.L_platform(lang)

    @property
    def L_title(self):
        return self.thread and self.thread.L_title

    @cached_property
    def link(self):
        return self.thread and url('thread', self.thread.uid)

    @property
    def merged_number(self):
        return None

    @cached_property
    def next_city(self):
        return settlement_or_station_title(self.rtstation.next_station)

    @cached_property
    def next_station(self):
        return self.rtstation.next_station

    @property
    def number(self):
        return self.thread.number

    @cached_property
    def prev_city(self):
        return settlement_or_station_title(self.rtstation.prev_station)

    @cached_property
    def prev_station(self):
        return self.rtstation.prev_station

    @property
    def real_event(self):
        return self.event

    @property
    def schedule(self):
        return self.rtstation and self.rtstation.schedule

    @property
    def station(self):
        return self.rtstation.station

    @property
    def subtype_code(self):
        return self.thread.subtype_code

    @property
    def t_model(self):
        return self.thread.t_model

    @property
    def t_type(self):
        return self.thread.t_type

    @property
    def thread(self):
        return self.rtstation.thread

    @property
    def event_planned_dt(self):
        return self.event_dt

    @cached_property
    def title(self):
        return self.thread and self.thread.title

    def get_popular_title(self, lang=None):
        return self.thread and self.thread.get_popular_title(lang=lang)

    @property
    def update(self):
        return self.thread.update


class ScheduleRoute(AbstractScheduleRoute):
    def __init__(self, schedule_item,
                 event_dt,
                 naive_start_dt, arrival_dt, departure_dt,
                 is_all_days_route=False):

        self.is_all_days_route = is_all_days_route
        self.schedule_item = schedule_item
        self.rtstation_id = schedule_item.rtstation.id
        self.rtstation = schedule_item.rtstation
        self.event = schedule_item.real_event
        self.is_last_station = schedule_item.is_last_station

        self.naive_start_dt = naive_start_dt

        self.event_dt = event_dt
        self.arrival_dt = arrival_dt
        self.departure_dt = departure_dt

        self.direction = None
        self.hidden = False

    @property
    def loc_datetime(self):
        warnings.warn("Use self.event_dt, was remarked at 2015-02-28", RaspDeprecationWarning, stacklevel=2)

        return self.event_dt

    @cached_property
    def start_date(self):
        return self.naive_start_dt.date()

    @cached_property
    def loc_arrival_datetime(self):
        return self.arrival_dt

    @cached_property
    def loc_arrival_t(self):
        return self.arrival_dt and self.arrival_dt.time()

    @cached_property
    def loc_departure_t(self):
        return self.departure_dt and self.departure_dt.time()

    @cached_property
    def loc_departure_datetime(self):
        return self.departure_dt

    @cached_property
    def mask(self):
        return self.thread.mask(environment.today()).shifted(self.mask_shift)

    @cached_property
    def mask_shift(self):
        return (self.event_dt.date() - self.naive_start_dt.date()).days


class ScheduleItem(object):
    def __init__(self, station, event, rowdict, out_tz):
        self.out_tz = out_tz
        self.start_tz = pytz.timezone(rowdict['thread__time_zone'])

        self.rts_time_zone = rowdict['time_zone']
        self.rts_tz = pytz.timezone(self.rts_time_zone)

        self.event = event or EVENT_DEPARTURE
        self.station = station
        self.rtstation_id = rowdict['id']

        self.tz_arrival = rowdict['tz_arrival']
        self.tz_departure = rowdict['tz_departure']

        self.is_fuzzy = rowdict['is_fuzzy']
        self.tz_thread_start_time = rowdict['thread__tz_start_time']
        self.year_days = rowdict['thread__year_days']
        self.schedule_plan_id = rowdict['thread__schedule_plan__id']

        self.real_event, self.is_last_station = (
            (EVENT_ARRIVAL, True)
            if self.tz_departure is None else
            (self.event, False)
        )
        real_event_offset_m = self.tz_arrival if self.real_event == EVENT_ARRIVAL else self.tz_departure

        self.tz_real_event_offset_m = real_event_offset_m

        self.real_event = self.real_event

        self.is_last_station = self.is_last_station

        self.rowdict = rowdict

        self.show_in_alldays_pages = rowdict['thread__show_in_alldays_pages']

    def __cmp__(self, other):
        return cmp(self.tz_event_time, other.tz_event_time)

    @cached_property
    def tz_event_time(self):
        dt = datetime.combine(_fake_date, self.tz_thread_start_time)

        dt += self.tz_real_event_timedelta

        return dt.time()

    @cached_property
    def tz_mask_shift(self):
        return (
            self.tz_thread_start_time.hour * 60 + self.tz_thread_start_time.minute +
            self.tz_real_event_offset_m
        ) / 1440

    @cached_property
    def tz_mask_shift_td(self):
        return timedelta(self.tz_mask_shift)

    def first_run(self, rts_tz_event_date):
        return RunMask.first_run(self.year_days, rts_tz_event_date - self.tz_mask_shift_td)

    def is_run_at(self, rts_tz_event_date):
        """
        Проверяет ходит ли рейс в указанную дату
        """
        mask_date = rts_tz_event_date - self.tz_mask_shift_td

        return RunMask.runs_at(self.year_days, mask_date)

    def is_run_at_range(self, start_date, end_date):
        start = start_date - self.tz_mask_shift_td
        end = end_date - self.tz_mask_shift_td

        start_position = RunMask.mask_day_index(start)
        end_position = RunMask.mask_day_index(end)

        if start.year == end.year:
            return '1' in self.year_days[start_position: end_position]

        return '1' in self.year_days[start_position:] or '1' in self.year_days[:end_position]

    @cache_method_result
    def arrival_dt(self, naive_start_dt):
        if self.tz_arrival is not None:
            return self.rts_tz.localize(naive_start_dt + self.tz_arrival_timedelta)

    @cache_method_result
    def departure_dt(self, naive_start_dt):
        if self.tz_departure is not None:
            return self.rts_tz.localize(naive_start_dt + self.tz_departure_timedelta)

    @cached_property
    def tz_arrival_timedelta(self):
        return None if self.tz_arrival is None else timedelta(minutes=self.tz_arrival)

    @cached_property
    def tz_departure_timedelta(self):
        return None if self.tz_departure is None else timedelta(minutes=self.tz_departure)

    @cached_property
    def tz_real_event_timedelta(self):
        if self.real_event == EVENT_ARRIVAL:
            return self.tz_arrival_timedelta
        elif self.real_event == EVENT_DEPARTURE:
            return self.tz_departure_timedelta

    def __repr__(self):
        return "<ScheduleItem rtstation=%s, msk_time=%s>" % (self.rtstation_id, self.tz_event_time)


class IntervalScheduleItem(object):
    def __init__(self, station, event, rowdict, out_tz=None):
        """
        Считаем, что интервальные рейсы, всегда в локальном времени, и не обращаем внимания на временну зону
        """
        self.begin_time = rowdict['thread__begin_time']
        self.end_time = rowdict['thread__end_time']

        self.event = event or EVENT_DEPARTURE
        self.station = station

        self.rtstation_id = rowdict['id']
        self.is_fuzzy = rowdict['is_fuzzy']
        self.year_days = rowdict['thread__year_days']

        self.schedule_plan_id = rowdict['thread__schedule_plan__id']

        self.tz_arrival = rowdict['tz_arrival']
        self.tz_departure = rowdict['tz_departure']

        self.real_event, self.is_last_station = (
            (EVENT_ARRIVAL, True)
            if self.tz_departure is None else
            (self.event, False)
        )
        real_event_offset_m = self.tz_arrival if self.real_event == EVENT_ARRIVAL else self.tz_departure

        self.tz_real_event_offset_m = real_event_offset_m

        self.real_event = self.real_event

        self.is_last_station = self.is_last_station

        self.rowdict = rowdict

        self.show_in_alldays_pages = rowdict['thread__show_in_alldays_pages']

    def is_run_at(self, start_date):
        return RunMask.runs_at(self.year_days, start_date)

    def __cmp__(self, other):
        return cmp((self.begin_time, self.end_time), (other.begin_time, other.end_time))

    def __repr__(self):
        return "<ScheduleItem rtstation=%s, msk_time=%s>" % (self.rtstation_id, self.tz_event_time)


class IntervalScheduleRoute(AbstractScheduleRoute):
    def __init__(self, schedule_item, start_date, is_all_days_route=False):

        self.start_date = start_date
        self.is_all_days_route = is_all_days_route
        self.schedule_item = schedule_item
        self.rtstation_id = schedule_item.rtstation.id
        self.rtstation = schedule_item.rtstation
        self.event = schedule_item.real_event
        self.is_last_station = schedule_item.is_last_station

        self.hidden = False
        self.mask_shift = 0

        self.begin_time = schedule_item.begin_time
        self.end_time = schedule_item.end_time

    def L_days_text(self, *args, **kwargs):
        kwargs['thread_start_date'] = self.start_date

        return self.thread and self.thread.L_days_text(self.mask_shift, *args, **kwargs)
