# -*- coding: utf-8 -*-

import bisect
import heapq
import logging
import calendar
import warnings
from itertools import cycle
from datetime import datetime, time, timedelta

import pytz
from django.contrib.admin.utils import get_fields_from_path
from django.db.models import Q, F
from django.utils.functional import cached_property

from travel.avia.library.python.common.utils.warnings import RaspDeprecationWarning
from travel.avia.library.python.common.models.geo import Station
from travel.avia.library.python.common.models.schedule import (
    RThread, RTStation, RThreadType, ScheduleExclusion, TrainSchedulePlan, StationSchedule
)
from travel.avia.library.python.common.models.transport import TransportType
from travel.avia.library.python.common.models_utils import fetch_related_safe
from travel.avia.library.python.common.utils.caching import cache_method_result
from travel.avia.library.python.common.utils import environment

from travel.avia.library.python.stationschedule.models import ScheduleRoute, ScheduleItem, IntervalScheduleItem, IntervalScheduleRoute
from travel.avia.library.python.stationschedule.utils import add_schedule_local_title


log = logging.getLogger(__name__)

# Лимит 1500 подходит, т.к. в расписании по станции у поездов должны влезать все.
# На июль 2014 максимальное количество ниток, проходящих через одну ЖД станцию ~1000
# Все самолеты начиная с 3.0 будут отбражаться в режиме statuses или real, у них есть слайдер.
DEFAULT_ITEMS_NUMBER_LIMIT = 1500
PREV_DAYS = 3
NEXT_DAYS = 270


def peek_values(qs, field_names):
    """
        "Подсматривает" значения указанных полей в QuerySet, добавляя их
        в запрос. Это позволяет получать необходимые данные за один
        минимальный запрос к базе.
    """
    field_aliases = {}
    select_query = {}

    for field_name in field_names:
        fields = get_fields_from_path(qs.model, field_name)

        if len(fields) == 1:
            continue

        model = fields[-2].rel.to
        model_field = fields[-1]
        model_field_name = model_field.db_column or model_field.name

        if model_field is model._meta.pk:
            model = fields[-3].rel.to
            model_field_name = '{}_{}'.format(fields[-2].name, model_field_name)

        field_alias = '_alias{}'.format(len(select_query))

        select_query[field_alias] = '`{}`.`{}`'.format(model._meta.db_table,
                                                       model_field_name)

        field_aliases[field_name] = field_alias

    for obj in qs.extra(select=select_query):
        yield (
            obj,
            {
                field_name:
                    getattr(obj, field_aliases.get(field_name, field_name))
                for field_name in field_names
            }
        )


class BaseAbstractSchedule(object):
    """
    Расписание по станции

    station       - станция
    event         - 'arrival' или 'departure'
    schedule_date - дата, на которую необходимо получить расписание. None - на все дни
    items_number_limit - максимальное кол-во рейсов, для которых необходимо построить расписание
    start_time_limit - время (без даты, местн.) начиная с которого надо показывать расписание (напрмер с '13:00')
    end_time_limit   - время (без даты, местн.) до которого надо показывать расписание (напрмер до '15:00')
    start_datetime_limit - время с датой (местн.) начиная с которого надо показывать расписание (напрмер с
                           '2014-03-01 13:00'). Если указан schedule_date, то игнорируется.
    end_datetime_limit   - время с датой (местн.) начиная с которого надо показывать расписание (напрмер до
                           '2014-03-07 13:00'). Если указан schedule_date, то игнорируется.
    with_interval - True - включать интервальные рейсы, False - не показывать интервальные рейсы
    t_type_code   - тип транспорта, может быть строкой, либо списком строк
    take_fuzzy    - True - в результат не попадают рейсы с нечеткое время отправления/прибытия, False - в результате
                    будут рейсы как с четким так и нечетким временем отправления/прибытия
    terminal      - дополнительная фильтрация по терминалу аэропорта

    kwargs        - служебный параметр для игнорирования неиспользуемых именованных параметров
    """

    DefaultScheduleItemClass = ScheduleItem
    DefaultScheduleRouteClass = ScheduleRoute
    FETCHING_FIELDS = None

    def __init__(self, station, event='departure', schedule_date=None,
                 limit=DEFAULT_ITEMS_NUMBER_LIMIT, offset=0,
                 start_time_limit=None, end_time_limit=None,
                 start_datetime_limit=None, end_datetime_limit=None,
                 t_type_code=None, take_fuzzy=True, terminal=None,
                 all_days_next_days=NEXT_DAYS, all_days_prev_days=PREV_DAYS,
                 **kwargs):

        self.out_tz = station.pytz

        # Количестов рейсов, без post_filters
        self._amount = None

        # Количество рейсов без лимитов, если такое возможно
        self._total = None

        self.today = environment.today()

        self.station = station

        self.event = event

        self.t_type_code = t_type_code

        self.offset = offset
        self.limit = limit
        self.schedule_date = schedule_date

        self.start_time_limit = start_time_limit
        self.end_time_limit = end_time_limit

        self.start_datetime_limit = start_datetime_limit
        self.end_datetime_limit = end_datetime_limit

        self.take_fuzzy = take_fuzzy
        self.terminal = terminal

        self.limit_is_reached = False

        # Filters
        self._schedule_items_with_time_filters = []
        self._raw_schedule_route_filters = []
        self._schedule_route_filters = []
        self._post_filters = []

        self._schedule_items = None
        self._schedule_routes = None

        self.schedule_item_class = kwargs.pop('schedule_item_class', self.DefaultScheduleItemClass)
        self.schedule_route_class = kwargs.pop('schedule_route_class', self.DefaultScheduleRouteClass)

        self.all_days_next_days = all_days_next_days
        self.all_days_prev_days = all_days_prev_days

        if kwargs:
            raise ValueError("Unhandled kwargs: %r" % kwargs)

    def calc_datetime_limits(self, **kwargs):

        self.schedule_date = kwargs.get('schedule_date', self.schedule_date)

        self.start_time_limit = kwargs.get('start_time_limit', self.start_time_limit)
        self.end_time_limit = kwargs.get('end_time_limit', self.end_time_limit)

        self.start_datetime_limit = kwargs.get('start_datetime_limit', self.start_datetime_limit)
        self.end_datetime_limit = kwargs.get('end_datetime_limit', self.end_datetime_limit)

        has_dt_limits = (self.schedule_date or self.start_datetime_limit or self.end_datetime_limit)
        self._all_day_schedule = not has_dt_limits

        if self._all_day_schedule:
            self.start_time_limit = self.start_time_limit or time.min
            self.end_time_limit = self.end_time_limit or time.max

            self.start_datetime_limit = datetime.combine(self.today, self.start_time_limit)
            self.end_datetime_limit = datetime.combine(self.today, self.end_time_limit)

        elif self.schedule_date:
            self.start_datetime_limit = self._default_start_datetime_limit(self.schedule_date)
            self.end_datetime_limit = self._default_end_datetime_limit(self.schedule_date)

        else:
            self.start_datetime_limit = self.start_datetime_limit
            self.end_datetime_limit = self.end_datetime_limit

        # Переводим время во временную зону станции
        self.start_datetime_limit = self._local_datetime(self.start_datetime_limit)
        self.end_datetime_limit = self._local_datetime(self.end_datetime_limit)

    def _default_start_datetime_limit(self, schedule_date):
        """Возвращает время с котрого необходимо построить расписание при построении расписания на конкретную дату"""
        return datetime.combine(schedule_date, time.min)

    def _default_end_datetime_limit(self, schedule_date):
        """Возвращает время до котрого необходимо построить расписание при построении расписания на конкретную дату"""
        return datetime.combine(schedule_date, time.max)

    def _local_datetime(self, dt):
        if not getattr(dt, 'tzinfo', None):
            return self.station.pytz.localize(dt)

        return dt.astimezone(self.station.pytz)

    @property
    def schedule_routes(self):
        """
        Возвращает расписание по станции
        """

        return self.build_schedule_routes()

    @property
    def amount(self):
        # Строим расписание
        self.build_schedule_routes()

        return self._amount

    @property
    def total(self):
        # Строим расписание
        self.build_schedule_routes()

        return self._total

    def build_schedule_routes(self):
        if self._schedule_routes:
            return self._schedule_routes

        self._schedule_routes = self._build_schedule_routes()

        return self._schedule_routes

    def _build_schedule_routes(self):
        raise NotImplementedError()

    def get_fetching_fields(self):
        return self.FETCHING_FIELDS

    def build_rtstations_query(self):
        qs = (
            RTStation.objects
                     .filter(station=self.station)
                     .filter(in_station_schedule=True, is_technical_stop=False)
                     .filter(
                         (~Q(tz_departure=F('tz_arrival'))) |
                         Q(tz_departure__isnull=True) |
                         Q(tz_arrival__isnull=True))
                     .filter(thread__route__hidden=False)

                     .order_by()
        )

        if self.t_type_code:
            if isinstance(self.t_type_code, basestring):
                t_type_codes = set([self.t_type_code])
            else:
                t_type_codes = set(self.t_type_code)

            if any(t_type_code in t_type_codes for t_type_code in TransportType.WATER_TTYPE_CODES):
                t_type_codes |= set(TransportType.WATER_TTYPE_CODES)

            t_type_pks = [TransportType.objects.get(code=c).id for c in t_type_codes]

            qs = qs.filter(thread__t_type__in=t_type_pks)

        if not self.take_fuzzy:
            qs = qs.filter(is_fuzzy=False)

        if self.terminal:
            qs = qs.filter(terminal=self.terminal)

        if self._excluded_threads:
            qs = qs.exclude(thread__in=self._excluded_threads)

        return qs

    def iter_rtstations_fields(self):
        return peek_values(self.build_rtstations_query(),
                           self.get_fetching_fields())

    @cache_method_result
    def build_schedule_items(self):
        ScheduleItem = self.schedule_item_class
        result = []

        for rtstation, rowdict in self.iter_rtstations_fields():
            schedule_item = ScheduleItem(self.station, self.event,
                                         rowdict, self.out_tz)

            rtstation.station = self.station
            schedule_item.rtstation = rtstation

            if (schedule_item.tz_real_event_offset_m is not None and
                    not self._is_fuzzy_departure_item(schedule_item)):
                # Пропускаем рейсы с нечетким временем отправления с данной станции
                result.append(schedule_item)

        return result

    @property
    def schedule_items(self):
        return self.build_schedule_items()

    @cached_property
    def _excluded_threads(self):
        return set(t.id for t in ScheduleExclusion.get_excluded_thread_for_station(self.station))

    def _is_fuzzy_departure_item(self, schedule_item):
        return schedule_item.real_event == 'departure' and schedule_item.is_fuzzy


class AbstractSchedule(BaseAbstractSchedule):
    FETCHING_FIELDS = [
        'id', 'time_zone', 'tz_arrival', 'tz_departure',
        'is_fuzzy', 'thread__tz_start_time', 'thread__time_zone', 'thread__year_days',
        'thread__schedule_plan__id', 'thread__number', 'thread__show_in_alldays_pages'
    ]

    @property
    def schedule(self):
        warnings.warn('Use RunMask.first_run, was remarked at 2015-02-28', RaspDeprecationWarning, stacklevel=2)
        return self.schedule_routes

    def build(self, **kwargs):
        self.reset_filters()

        self.calc_datetime_limits(**kwargs)

        self.build_schedule_items()

        self._schedule_routes = None
        self._total = None
        self._amount = None

        self.add_filters(**kwargs)

        return self

    def _build_schedule_routes(self):
        has_schedule_route_filter = bool(self._schedule_route_filters)

        schedule_routes_gen = self.gen_schedule()
        schedule_routes_gen = self.apply_filters(schedule_routes_gen, 'raw_schedule_route_filters')

        if has_schedule_route_filter:
            pre_schedule_routes = list(schedule_routes_gen)
            self.fill_schedule_routes(pre_schedule_routes)

            schedule_routes_gen = self.apply_filters(pre_schedule_routes, 'schedule_route_filters')

        schedule_routes = self.apply_limits(schedule_routes_gen)

        if not has_schedule_route_filter:
            self.fill_schedule_routes(schedule_routes)

        self._amount = len(schedule_routes)

        schedule_routes = list(self.apply_filters(schedule_routes, 'post_filters'))

        return schedule_routes

    def add_filters(self, **kwargs):
        if self.event:
            event = self.event

            def skip_routes_with_last_station(si_with_msk_dt_get):
                for schedule_item, msk_dt in si_with_msk_dt_get:
                    if event == 'departure' and schedule_item.is_last_station:
                        continue

                    yield schedule_item, msk_dt

            self.add_schedule_items_with_time_filter(skip_routes_with_last_station)

        if kwargs.get('schedule_plan'):
            schedule_plan_id = kwargs.get('schedule_plan').id

            def filter_by_schedule_plan(si_with_msk_dt_get):
                for schedule_item, msk_dt in si_with_msk_dt_get:
                    if schedule_item.schedule_plan_id not in (None, schedule_plan_id):
                        continue

                    yield schedule_item, msk_dt

            self.add_schedule_items_with_time_filter(filter_by_schedule_plan)

    def apply_filters(self, generator, filters_name):
        for filter_func in getattr(self, '_{}'.format(filters_name)):
            generator = filter_func(generator)

        return generator

    def reset_filters(self):
        self._raw_schedule_route_filters = []
        self._schedule_items_with_time_filters = []

    def add_schedule_items_with_time_filter(self, filter_func):
        self._schedule_items_with_time_filters.append(filter_func)

    def add_raw_schedule_route_filter(self, filter_func):
        self._raw_schedule_route_filters.append(filter_func)

    def add_schedule_route_filter(self, filter_func):
        self._schedule_route_filters.append(filter_func)

    def add_schedule_route_post_filter(self, filter_func):
        """
        Фильтрация применяется уже после применения лимитов по количеству
        """
        self._post_filters.append(filter_func)

    @cached_property
    def hited_schedule_plans_id(self):
        return {si.schedule_plan_id for si in self.schedule_items}

    @cache_method_result
    def current_next_plans(self, today):
        plans_qs = TrainSchedulePlan.objects.filter(id__in=self.hited_schedule_plans_id)

        return TrainSchedulePlan.get_current_and_next(today, qs=plans_qs)

    def apply_limits(self, schedule_routes_gen):
        schedule_routes = []

        offset = self.offset
        limit = self.limit

        total_count = 0
        collected = 0
        done = False

        for index, sr in enumerate(schedule_routes_gen):
            total_count += 1

            if done:
                continue

            if index < offset:
                continue

            schedule_routes.append(sr)
            collected += 1

            if collected == limit:
                done = True

        self._total = total_count

        return schedule_routes

    def gen_schedule_items(self):
        return self.gen_items_event_datetime()

    def gen_schedule(self):
        items_gen = self.gen_schedule_items()

        items_gen = self.apply_filters(items_gen, 'schedule_items_with_time_filters')

        for schedule_item, rts_tz_event_dt in items_gen:
            yield self.build_raw_schedule_route(schedule_item, rts_tz_event_dt)

    def build_raw_schedule_route(self, schedule_item, rts_tz_event_dt):
        ScheduleRoute = self.schedule_route_class

        if self._all_day_schedule:
            nearest_run_day = schedule_item.first_run(rts_tz_event_dt.date()) or rts_tz_event_dt.date()
            naive_start_dt = datetime.combine(nearest_run_day, schedule_item.tz_thread_start_time)

            naive_rts_tz_event_dt = naive_start_dt + schedule_item.tz_real_event_timedelta
            rts_tz_event_dt = schedule_item.rts_tz.localize(naive_rts_tz_event_dt)
        else:
            naive_rts_tz_event_dt = rts_tz_event_dt.replace(tzinfo=None)
            naive_start_dt = naive_rts_tz_event_dt - schedule_item.tz_real_event_timedelta

        event_dt = rts_tz_event_dt.astimezone(self.out_tz)

        arrival_dt = schedule_item.arrival_dt(naive_start_dt)
        arrival_dt = arrival_dt and arrival_dt.astimezone(self.out_tz)

        departure_dt = schedule_item.departure_dt(naive_start_dt)
        departure_dt = departure_dt and departure_dt.astimezone(self.out_tz)

        return ScheduleRoute(schedule_item,
                             event_dt,

                             naive_start_dt,
                             arrival_dt,
                             departure_dt,

                             is_all_days_route=self._all_day_schedule)

    def build_rtstations_query(self):
        qs = super(AbstractSchedule, self).build_rtstations_query()

        return (
            qs.exclude(thread__type=RThreadType.THROUGH_TRAIN_ID)
              .exclude(thread__type=RThreadType.INTERVAL_ID)
              .exclude(thread__type=RThreadType.CANCEL_ID)
        )

    @cached_property
    def schedule_items_by_tz(self):
        schedule_items = self.schedule_items

        schedule_items_by_tz = {}
        for si in schedule_items:
            schedule_items_by_tz.setdefault(si.rts_time_zone, []).append(si)

        for si_list in schedule_items_by_tz.values():
            si_list.sort(key=lambda si: si.tz_event_time)

        return schedule_items_by_tz

    @property
    def schedule_items_count(self):
        return len(self.schedule_items)

    def fill_schedule_routes(self, schedule_routes):
        rtstations = set(s.rtstation for s in schedule_routes if s.rtstation)

        schedules = dict((s.rtstation_id, s) for s in StationSchedule.objects.filter(rtstation__in=rtstations))

        for rts in rtstations:
            rts.schedule = schedules.get(rts.id, StationSchedule())

        fetch_related_safe(rtstations, 'thread', 'next_station', 'prev_station', model=RTStation)
        fetch_related_safe([rts.thread for rts in rtstations], 'supplier', 'express_lite', 'company', model=RThread)
        fetch_related_safe([rts.next_station for rts in rtstations if rts.next_station] +
                           [rts.prev_station for rts in rtstations if rts.prev_station], 'settlement', model=Station)

        TrainSchedulePlan.fill_thread_plans([sr.thread for sr in schedule_routes])

        add_schedule_local_title(schedule_routes)

    @cached_property
    def _all_day_mask_start(self):
        return self.today - timedelta(self.all_days_prev_days)

    @cached_property
    def _all_day_mask_end(self):
        return self.today + timedelta(self.all_days_next_days)

    def gen_items_event_datetime(self):
        schedule_items_by_tz = self.schedule_items_by_tz

        if not schedule_items_by_tz:
            return

        generators = []

        for tz in schedule_items_by_tz.keys():
            generators.append(self.gen_items_with_datetime_within_single_rts_tz_datetime(tz))

        for rts_tz_event_dt, schedule_item in heapq.merge(*generators):
            # Переворачиваем для совместимости с остальными участками кода
            yield schedule_item, rts_tz_event_dt

    def gen_items_with_datetime_within_single_rts_tz_datetime(self, time_zone):
        """
        Генерируем schedule_items и время старта по Москве
        Для расписания на дату, генерируются только проходящие в этот день рейсы

        Массив schedule_items получается следующим образом:

            Если msk_start_time 01:00 по Москве получаем следующий массив:
                06:00
                13:00
                18:00
                msk_switch_date

            Если msk_start_time 08:00 по Москве получаем следующий массив:
                13:00
                18:00
                msk_switch_date
                06:00

            Если msk_start_time 20:00 по Москве получаем следующий массив:
                msk_switch_date
                06:00
                13:00
                18:00

        В результате преобразования массива, мы сразу же получаем результаты, где
        msk_datetime(время события) будет возрастать начиная с msk_start_datetime_limit

        """

        # TODO:нужно сделать для этого метода тест.

        schedule_items_by_tz = self.schedule_items_by_tz

        if not schedule_items_by_tz:
            return

        schedule_items = schedule_items_by_tz[time_zone]

        # Метка для переключения даты
        date_switch = object()

        end_datetime_limit = self.end_datetime_limit

        rts_tz = pytz.timezone(time_zone)

        rts_tz_start_datetime_limit = self.start_datetime_limit.astimezone(rts_tz)

        rts_tz_start_time = rts_tz_start_datetime_limit.time()
        rts_tz_start_date = rts_tz_start_datetime_limit.date()

        # Перестраиваем сортированый по времени массив так,
        # чтобы в начале массива время совпадало с self.msk_start_datetime_limit.time()
        start_from_index = bisect.bisect_left([si.tz_event_time for si in schedule_items],
                                              rts_tz_start_time)

        schedule_items = schedule_items[start_from_index:] + [date_switch] + schedule_items[:start_from_index]

        for schedule_item in cycle(schedule_items):
            if schedule_item is date_switch:
                rts_tz_start_date += timedelta(1)
                continue

            rts_tz_event_dt = rts_tz.localize(datetime.combine(rts_tz_start_date, schedule_item.tz_event_time))

            if rts_tz_event_dt > end_datetime_limit:
                return

            if self._all_day_schedule:
                if not schedule_item.show_in_alldays_pages:
                    continue
                if schedule_item.is_run_at_range(self._all_day_mask_start, self._all_day_mask_end):
                    yield rts_tz_event_dt, schedule_item

            elif schedule_item.is_run_at(rts_tz_start_date):
                yield rts_tz_event_dt, schedule_item

    def reverse_gen_items_event_datetime(self):
        schedule_items_by_tz = self.schedule_items_by_tz

        if not schedule_items_by_tz:
            return

        generators = []

        for tz in schedule_items_by_tz.keys():
            generators.append(self.reverse_gen_items_with_datetime_within_single_rts_tz_datetime(tz))

        for _mts, rts_tz_event_dt, schedule_item in heapq.merge(*generators):
            # Переворачиваем для совместимости с остальными участками кода
            yield schedule_item, rts_tz_event_dt

    def reverse_gen_items_with_datetime_within_single_rts_tz_datetime(self, time_zone):
        """
        Тоже самое что и gen_items_with_datetime_within_single_rts_tz_datetime,
        только по убыванию в обратном порядке, начиная с < msk_start_datetime_limit
        максимум на 7 дней
        """
        schedule_items_by_tz = self.schedule_items_by_tz

        if not schedule_items_by_tz:
            return

        schedule_items = schedule_items_by_tz[time_zone]

        # Метка для переключения даты
        date_switch = object()

        rts_tz = pytz.timezone(time_zone)

        rts_tz_start_datetime_limit = self.start_datetime_limit.astimezone(rts_tz)

        rts_tz_start_time = rts_tz_start_datetime_limit.time()
        rts_tz_start_date = rts_tz_start_datetime_limit.date()

        # Перестраиваем сортированый по времени массив так,
        # чтобы в начале массива время совпадало с self.msk_start_datetime_limit.time()
        start_from_index = bisect.bisect_left([si.tz_event_time for si in schedule_items],
                                              rts_tz_start_time)

        schedule_items = schedule_items[start_from_index:] + [date_switch] + schedule_items[:start_from_index]
        schedule_items = schedule_items[::-1]

        last_check_date = rts_tz_start_date - timedelta(7)

        for schedule_item in cycle(schedule_items):
            if schedule_item is date_switch:
                rts_tz_start_date -= timedelta(1)
                continue

            rts_tz_event_dt = rts_tz.localize(datetime.combine(rts_tz_start_date, schedule_item.tz_event_time))

            if rts_tz_start_date < last_check_date:
                return

            ts = calendar.timegm(rts_tz_event_dt.utctimetuple())

            if self._all_day_schedule:
                if schedule_item.is_run_at_range(self._all_day_mask_start, self._all_day_mask_end):

                    # Что правильно сортировалось heap.merge отдаем отрицательный timestamp
                    yield -ts, rts_tz_event_dt, schedule_item

            elif schedule_item.is_run_at(rts_tz_start_date):
                # Что правильно сортировалось heap.merge отдаем отрицательный timestamp
                yield -ts, rts_tz_event_dt, schedule_item


class BaseIntervalSchedule(BaseAbstractSchedule):
    DefaultScheduleItemClass = IntervalScheduleItem
    DefaultScheduleRouteClass = IntervalScheduleRoute

    FETCHING_FIELDS = [
        'id', 'tz_arrival', 'tz_departure',
        'is_fuzzy',

        'thread__begin_time', 'thread__end_time',
        'thread__year_days',
        'thread__schedule_plan__id', 'thread__number', 'thread__show_in_alldays_pages'
    ]

    def build_rtstations_query(self):
        qs = super(BaseIntervalSchedule, self).build_rtstations_query()

        return qs.filter(thread__type=RThreadType.INTERVAL_ID)

    def build_schedule_items(self):
        schedule_items = super(BaseIntervalSchedule, self).build_schedule_items()
        schedule_items.sort()

        return schedule_items

    def build(self, **kwargs):
        self.calc_datetime_limits(**kwargs)

        self._schedule_routes = None
        self._amount = None
        self._total = None

        self.build_schedule_routes()

        return self

    def _build_schedule_routes(self):
        IntervalScheduleRoute = self.schedule_route_class

        start_time_limit = self.start_datetime_limit.time()
        end_time_limit = self.end_datetime_limit.time()

        start_day = self.start_datetime_limit.date()
        end_day = self.end_datetime_limit.date()
        day = start_day
        event = self.event

        schedule_routes = []
        while day <= self.end_datetime_limit.date():
            for si in self.schedule_items:
                if event == 'departure' and si.is_last_station:
                    continue

                # Убираем нитки не пересекающиеся с нашим диапазоном
                if day == end_day and si.begin_time > end_time_limit:
                    continue

                elif day == start_day and si.end_time < start_time_limit:
                    continue

                if self._all_day_schedule:
                    if not si.show_in_alldays_pages:
                        continue
                else:
                    if not si.is_run_at(day):
                        continue

                schedule_routes.append(IntervalScheduleRoute(si, day, is_all_days_route=self._all_day_schedule))

            day += timedelta(1)

        self._total = len(schedule_routes)

        schedule_routes = schedule_routes[self.offset:self.offset + self.limit]

        self._amount = len(schedule_routes)

        return schedule_routes
