# -*- coding: utf-8 -*-

from django.utils.functional import cached_property

from travel.avia.library.python.common.utils import environment
from travel.avia.library.python.common.utils.date import MSK_TZ
from travel.avia.library.python.route_search.by_number.helpers import number_variants

from travel.avia.library.python.stationschedule.type import AbstractSchedule
from travel.avia.library.python.stationschedule.utils import tablo_status


class TabloStatusesSchedule(AbstractSchedule):
    def __init__(self, station, **kwargs):
        self.query = (kwargs.pop('query', u'') or u'').strip()
        self.one_earlier = kwargs.pop('one_earlier', False)

        # Если мы успели набрали минимум min_items_number_limit рейсов до
        # min_datetime_limit, то останавливаем выборку.
        # Иначе пытаемся добрать до min_items_number_limit, ограничиваясь уже end_time_limit
        self.min_datetime_limit = kwargs.pop('min_datetime_limit', None)
        self.min_items_number_limit = kwargs.pop('min_items_number_limit', None)
        self.max_items_number_limit = kwargs.pop('max_items_number_limit', None)

        super(TabloStatusesSchedule, self).__init__(station, **kwargs)

        self.max_items_number_limit = self.max_items_number_limit or self.limit

        self._schedule_filtered = []

    def add_filters(self, **kwargs):
        super(TabloStatusesSchedule, self).add_filters(**kwargs)

        if self.query:
            def query_filter(schedule_route_gen):
                for schedule_route in schedule_route_gen:
                    if self._like_query(schedule_route):
                        yield schedule_route

            self.add_schedule_route_post_filter(query_filter)

    def apply_limits(self, schedule_routes_gen):
        if self.min_datetime_limit:
            min_datetime_limit = self._local_datetime(self.min_datetime_limit)
        else:
            min_datetime_limit = None

        schedule_routes = []

        collected = 0

        min_items_number_limit = self.min_items_number_limit
        max_items_number_limit = self.max_items_number_limit

        for sr in schedule_routes_gen:
            collected += 1
            schedule_routes.append(sr)

            if min_datetime_limit:
                if sr.event_dt <= min_datetime_limit:
                    continue

                elif min_items_number_limit and collected >= min_items_number_limit:
                    break

            if max_items_number_limit and collected >= max_items_number_limit:
                break

        return schedule_routes

    def fill_schedule_routes(self, schedule_routes):
        super(TabloStatusesSchedule, self).fill_schedule_routes(schedule_routes)

        self._fill_statuses(schedule_routes)

    @cached_property
    def slider_width(self):
        if self.min_datetime_limit:
            if len(self.schedule_routes) >= self.min_items_number_limit:
                return self.schedule_routes[-1].event_dt - self.start_datetime_limit

        return self.end_datetime_limit - self.start_datetime_limit

    def gen_schedule_items(self):
        super_gen = super(TabloStatusesSchedule, self).gen_schedule_items()

        need_to_add_earlier = self.one_earlier and not self.query

        if need_to_add_earlier:
            try:
                yield self.reverse_gen_items_event_datetime().next()
            except StopIteration:
                pass

        for i in super_gen:
            yield i

    def _fill_statuses(self, schedule_routes):
        now = MSK_TZ.localize(environment.now())

        for schedule_route in schedule_routes:
            schedule_route.status_code, schedule_route.status = tablo_status(
                None,
                schedule_route.event_dt,
                schedule_route.event_dt,
                False,
                self.event,
                now,
                now,
                self.station
            )

    def _like_query(self, schedule_route):
        for field_value in self._search_field_values(schedule_route):
            try:
                if field_value.contains(self.query):
                    return True

            except AttributeError:
                if self.query in field_value:
                    return True

        return False

    def _search_field_values(self, schedule_route):
        if schedule_route.number:
            yield schedule_route.number

        yield schedule_route.L_title()

        yield u"%s %s" % (schedule_route.number, schedule_route.L_title())

        next_station = schedule_route.rtstation.next_station

        if next_station:
            yield next_station.L_title

        prev_station = schedule_route.rtstation.prev_station

        if prev_station:
            yield prev_station.L_title

        if schedule_route.company:
            yield schedule_route.company.L_title

        if next_station and next_station.country:
            yield next_station.country.L_title

        for number in number_variants(schedule_route.number):
            yield number
