# coding: utf-8
from __future__ import unicode_literals

import logging
from collections import namedtuple
from datetime import date, datetime
from typing import Any, Optional, Protocol

import ujson

from travel.avia.library.python.ticket_daemon.memo import memoize, CacheWithKeyTTL, CacheInMemcache
from travel.avia.library.python.shared_flights_client.client import OperatingFlight, SharedFlightsClient

logger = logging.getLogger(__name__)


def _operating_flight_serializer(v):
    # type: (OperatingFlight)->str
    return ujson.dumps(v)


def _operating_flight_deserializer(v):
    # type: (str)->Optional[OperatingFlight]
    j = ujson.loads(v)
    if j is None:
        return j
    return OperatingFlight(*j)


class _Cache(Protocol):
    def set(self, key, value, timeout=None):
        # type: (str, Any, float)->None
        pass

    def get(self, key):
        # type: (str)->Any
        pass


GetOperatingFlightReq = namedtuple('GetOperatingFlightReq', 'company_code number departure_day')


class OperatingFlightProvider(object):
    def __init__(self, client, daemon_cache):
        # type: (SharedFlightsClient, _Cache)->None

        def _daemon_cache_operating_flight_keyfun(*args):
            return ''.join(map(str, args))

        # memoize _get_operating_flight method
        def _get_operating_flight(company_code, number, departure_day):
            return client.get_operating_flight(company_code, number, departure_day)

        local_cache = memoize(
            keyfun=(lambda company_code, number, departure_day: (company_code, number, departure_day)),
            cache=CacheWithKeyTTL(60 * 60),  # default to an hour, this should be pretty static
        )
        shared_cache = memoize(
            keyfun=_daemon_cache_operating_flight_keyfun,
            cache=CacheInMemcache(
                mc=daemon_cache,
                ttl=3 * 60 * 60,  # 3 hours of remote cache ttl
                prefix='get_operating_flight',
                suffix='shared_v2',
                no_reset=True,
                serializer=_operating_flight_serializer,
                deserializer=_operating_flight_deserializer,
            ),
        )
        self._get_operating_flight = local_cache(shared_cache(_get_operating_flight))

    @staticmethod
    def map_segment_to_params(segment):
        company_code, number = segment['number'].split(' ')
        departure_day = datetime.strptime(segment['departure']['local'].split('T')[0], '%Y-%m-%d').date()
        return GetOperatingFlightReq(company_code, number, departure_day)

    def get_operating_flight(self, company_code, number, departure_day):
        # type: (str, str, date)->Optional[OperatingFlight]
        return self._get_operating_flight(company_code, number, departure_day)
