# -*- coding: utf-8 -*-
from __future__ import absolute_import

import time
import datetime
from abc import abstractmethod, abstractproperty, ABCMeta

import pytz

from travel.avia.library.python.common.utils.date import MSK_TZ
from travel.avia.library.python.ticket_daemon.memo import memoize

FORMAT_DATE = '%Y-%m-%d %H:%M:%S'
EPOCH = datetime.datetime(1970, 1, 1)
EPOCH_DATE = EPOCH.date()


def awarify(obj):
    if obj.tzinfo:
        return obj

    return MSK_TZ.localize(obj)


def get_utc_now():
    return pytz.UTC.localize(datetime.datetime.utcnow())


def get_msk_now():
    return get_utc_now().astimezone(MSK_TZ)


def unixtime():
    # type: () -> int
    return int(time.time())


def naive_datetime_to_timestamp(dt):
    # type: (datetime.datetime) -> int
    td = dt - EPOCH
    return int(td.total_seconds())


def naive_date_to_timestamp(d):
    # type: (datetime.date) -> int
    return int((d - EPOCH_DATE).total_seconds())


def aware_to_timestamp(localized_datetime):
    # type: (datetime.datetime) -> int
    """
        TODO: For working with pytz, it is recommended to call
         tz.localize(naive_dt) instead of naive_dt.replace(tzinfo=tz). dt.replace(tzinfo=tz)
         does not handle daylight savings time correctly.
        utc_naive = localized_datetime.astimezone(pytz.timezone('UTC')).replace(tzinfo=None)
    """
    utc_naive = localized_datetime.replace(
        tzinfo=None
    ) - localized_datetime.utcoffset()
    return naive_datetime_to_timestamp(utc_naive)


def aware_utc_offset(dt):
    # type: (datetime.datetime) -> int
    return int(dt.utcoffset().total_seconds())


class AbstractCachedDateTimeSerializer(object):
    __metaclass__ = ABCMeta
    FORMAT_DATE = abstractproperty()

    def __init__(self):
        self._cache = {}

    def serialize(self, dt):
        key = (dt, dt.tzinfo and dt.tzinfo.zone)
        if key not in self._cache:
            self._cache[key] = self.serialize_format(dt)

        return self._cache[key]

    @abstractmethod
    def serialize_format(self, dt):
        # type: (datetime.datetime) -> dict
        pass


class AbstractCachedDateTimeDeserializer(object):
    __metaclass__ = ABCMeta
    FORMAT_DATE = abstractproperty()

    def __init__(self):
        self._cache = {}

    def deserialize(self, raw_datetime):
        if raw_datetime not in self._cache:
            self._cache[raw_datetime] = datetime.datetime.strptime(raw_datetime, self.FORMAT_DATE)
        return self._cache[raw_datetime]


class DateTimeSerializer(AbstractCachedDateTimeSerializer):
    FORMAT_DATE = FORMAT_DATE

    def serialize_format(self, dt):
        # type: (datetime.datetime) -> dict
        return {
            'local_datetime': dt.strftime(self.FORMAT_DATE),
            'tzname': dt.tzinfo.zone,
            'offset': aware_utc_offset(dt) / 60
        }


class DateTimeSerializerV3(AbstractCachedDateTimeSerializer):
    FORMAT_DATE = '%Y-%m-%dT%H:%M:%S'

    def serialize_format(self, dt):
        return {
            'local': self.strftime(dt),
            'tzname': dt.tzinfo.zone,
            'offset': aware_utc_offset(dt) / 60
        }

    @staticmethod
    def strftime(dt, format=FORMAT_DATE):
        return dt.strftime(format)


class DateTimeDeserializerV3(AbstractCachedDateTimeDeserializer):
    FORMAT_DATE = DateTimeSerializerV3.FORMAT_DATE


class CachingDateUtils(object):
    def __init__(self):
        self.awarify = memoize(keyfun=lambda obj: obj)(awarify)
        self.get_utc_now = get_utc_now
        self.get_msk_now = get_msk_now
        self.unixtime = unixtime
        self.naive_datetime_to_timestamp = memoize(keyfun=lambda dt: dt)(naive_datetime_to_timestamp)
        self.naive_date_to_timestamp = memoize(keyfun=lambda dt: dt)(naive_date_to_timestamp)
        self.aware_to_timestamp = memoize(keyfun=lambda dt: dt)(aware_to_timestamp)
        self.aware_utc_offset = memoize(keyfun=lambda dt: dt)(aware_utc_offset)
