# -*- coding: utf-8 -*-
import logging
from itertools import islice, chain

from travel.avia.library.python.filter import skip_None_values
from travel.avia.library.python.ticket_daemon.protobuf_converting.big_wizard.filters_parser import FiltersParser
from travel.avia.library.python.ticket_daemon.protobuf_converting.big_wizard.search_result_converter import SearchResultConverter

log = logging.getLogger(__name__)  # Todo: one logger for module
search_result_converter = SearchResultConverter()
__all__ = ('search_result_filtering',)


class SearchResultFiltering(object):
    def filter_and_slice(self, search_result, params):
        """Изменяет переданный search_result согласно фильтрам из params

        :type search_result: ticket_daemon_lib.protobuf_converting.big_wizard.search_result_pb2.SearchResult
        :param dict params: {'filters': {...}, 'limit': <...>, 'offset': <...>}
        """
        if not search_result:
            return search_result

        fares = self._filter(
            search_result, search_result.fares, params.get('filters', {})
        )

        fares = self._slice(
            fares,
            limit=params.get('limit', None),
            offset=params.get('offset', 0),
        )

        self._remove_filtered(search_result, fares)

    def filter(self, search_result, params):
        new_fares = self._filter(
            search_result, search_result.fares, params
        )
        self._remove_filtered(search_result, new_fares)

    def slice(self, search_result, limit=None, offset=0):
        if not limit and not offset:
            return
        new_fares = self._slice(search_result.fares, limit, offset)
        self._remove_filtered(search_result, new_fares)

    @staticmethod
    def _filter(search_result, fares, params):
        filters = FiltersParser.parse_filters(skip_None_values(params))
        for _filter in filters:
            fares = _filter(fares, search_result)

        return fares

    @staticmethod
    def _slice(fares, limit, offset):
        if limit:
            return islice(fares, offset, offset + limit)
        else:
            return islice(fares, offset, None)

    @staticmethod
    def _remove_filtered(search_result, new_fares):
        """
        удаляет самолеты из flights, которых не осталось в 'fares'
        """
        if not isinstance(new_fares, list):
            fares = list(new_fares)
            if len(fares) == len(search_result.fares):
                return

            del search_result.fares[:]
            search_result.fares.extend(fares)

        routes = {
            route
            for fare in search_result.fares
            for route in chain(fare.route.forward, fare.route.backward)
        }

        for flight_key in search_result.flights.keys():
            if flight_key not in routes:
                del search_result.flights[flight_key]


search_result_filtering = SearchResultFiltering()
