# -*- coding: utf-8 -*-
from itertools import imap

from google.protobuf.json_format import MessageToDict

from travel.proto.avia.wizard.search_result_pb2 import (
    SearchResult, Flight, Fare, PollingStatus
)


class SearchResultConverter(object):
    def to_protobuf(self, search_result):
        """

        :type search_result: dict
        :rtype: SearchResult
        """

        flights_prepared_dict = dict()
        for flight_key, flight_description in search_result['flights'].iteritems():
            flights_prepared_dict[flight_key] = self._flight_to_protobuf(flight_description)

        prepared_fares = [self._fare_to_protobuf(x) for x in search_result['fares']]

        protobuf = SearchResult(
            qid=search_result['qid'],
            version=search_result['version'],
            flights=flights_prepared_dict,
            fares=prepared_fares,
            offers_count=search_result['offers_count'],
            polling_status=self._polling_status_to_protobuf(search_result.get('polling_status')),
        )

        return protobuf

    def to_dictionary(self, search_result):
        """

        :type search_result: SearchResult
        :rtype: dict
        """

        return {
            'qid': search_result.qid,
            'version': search_result.version,
            'flights': self._flights_from_protobuf(search_result.flights),
            'fares': self._fares_from_protobuf(search_result.fares),
            'offers_count': search_result.offers_count,
            'polling_status': self._polling_status_from_protobuf(search_result.polling_status)
        }

    def _fill_timestamp(self, local_timestamp_protobuf, local_timestamp):
        if local_timestamp:
            local_timestamp_protobuf.local = local_timestamp['local']
            local_timestamp_protobuf.tzname = local_timestamp['tzname']
            local_timestamp_protobuf.offset = local_timestamp['offset']

    def _flight_to_protobuf(self, flight):
        protobuf_flight = Flight()

        self._fill_timestamp(protobuf_flight.arrival, flight['arrival'])
        self._fill_timestamp(protobuf_flight.departure, flight['departure'])

        protobuf_flight.to_id = flight['to']
        protobuf_flight.from_id = flight['from']

        if flight['companyTariff'] is not None:
            protobuf_flight.company_tariff = flight['companyTariff']

        protobuf_flight.key = flight['key']

        if flight['company'] is not None:
            protobuf_flight.company = flight['company']
        if flight['aviaCompany'] is not None:
            protobuf_flight.avia_company = flight['aviaCompany']

        protobuf_flight.number = flight['number']

        return protobuf_flight

    def _fill_baggage(self, baggage_protobuf, baggage):
        baggage_protobuf.forward.extend([(x if x is not None else 'NNN') for x in baggage[0]])
        baggage_protobuf.backward.extend([(x if x is not None else 'NNN') for x in baggage[1]])

    def _fill_route(self, route_protobuf, route):
        route_protobuf.forward.extend(route[0])
        route_protobuf.backward.extend(route[1])

    def _fill_tariff(self, tariff_protobuf, tariff):
        tariff_protobuf.currency = tariff['currency']
        tariff_protobuf.value = tariff['value']

    def _fill_promo(self, promo_protobuf, promo):
        if not promo:
            return

        for key in ['code', 'end_ts']:
            value = promo.get(key)
            if value:
                setattr(promo_protobuf, key, value)

    def _fill_tariffs(self, tariffs_protobuf, tariffs):
        """

        :type tariffs_protobuf: ticket_daemon_lib.protobuf_converting.big_wizard.search_result_pb2.Tariffs
        """
        if not tariffs:
            return
        for key in ['with_baggage', 'without_baggage']:
            if key in tariffs:
                #  Assignment is not allowed in protocol message object.
                if key == 'with_baggage':
                    tariff_info = tariffs_protobuf.with_baggage
                elif key == 'without_baggage':
                    tariff_info = tariffs_protobuf.without_baggage

                self._fill_tariff(tariff_info.price, tariffs[key]['price'])
                tariff_info.partner = tariffs[key]['partner']
                tariff_info.conversion_partner = tariffs[key].get('conversion_partner', '')
                self._fill_baggage(tariff_info.baggage, tariffs[key]['baggage'])
                tariff_info.created_at = tariffs[key]['created_at']
                tariff_info.expire_at = tariffs[key]['expire_at']

    def _fare_to_protobuf(self, fare):
        protobuf_fare = Fare()

        if fare['charter'] is not None:
            protobuf_fare.charter = fare['charter']

        self._fill_promo(protobuf_fare.promo, fare.get('promo'))
        protobuf_fare.created_at = fare['created']
        self._fill_route(protobuf_fare.route, fare['route'])
        self._fill_baggage(protobuf_fare.baggage, fare['baggage'])
        protobuf_fare.expire_at = fare['expire']
        protobuf_fare.partner = fare['partner']
        protobuf_fare.conversion_partner = fare['conversion_partner']
        self._fill_tariff(protobuf_fare.tariff, fare['tariff'])
        protobuf_fare.popularity = fare['popularity']
        self._fill_tariffs(protobuf_fare.tariffs, fare.get('tariffs'))
        return protobuf_fare

    def _polling_status_to_protobuf(self, polling_status):
        if not polling_status:
            return

        protobuf_polling_status = PollingStatus()

        protobuf_polling_status.asked_partners.extend(polling_status['asked_partners'])
        protobuf_polling_status.asked_partners_count = polling_status['asked_partners_count']

        protobuf_polling_status.remaining_partners.extend(polling_status['remaining_partners'])
        protobuf_polling_status.remaining_partners_count = polling_status['remaining_partners_count']

        return protobuf_polling_status

    def _baggage_from_protobuf(self, baggage_protobuf):
        return [list(baggage_protobuf.forward), list(baggage_protobuf.backward)]

    def _route_from_protobuf(self, route_protobuf):
        return tuple(route_protobuf.forward), tuple(route_protobuf.backward)

    def _tariffs_from_protobuf(self, tariffs_protobuf):
        tariffs = MessageToDict(tariffs_protobuf, preserving_proto_field_name=True)
        for tariff in tariffs.itervalues():
            tariff['baggage'] = [
                tariff['baggage'].get('forward', []),
                tariff['baggage'].get('backward', [])
            ]
        return tariffs

    def _promo_from_protobuf(self, promo_protobuf):
        result = {}
        if not promo_protobuf or not promo_protobuf.code:
            return result

        result['code'] = promo_protobuf.code
        if promo_protobuf.end_ts:
            result['end_ts'] = promo_protobuf.end_ts

        return result

    def _fare_from_protobuf(self, fare_protobuf):
        return {
            'charter': fare_protobuf.charter,
            'created': fare_protobuf.created_at,
            'route': self._route_from_protobuf(fare_protobuf.route),
            'baggage': self._baggage_from_protobuf(fare_protobuf.baggage),
            'expire': fare_protobuf.expire_at,
            'partner': fare_protobuf.partner,
            'conversion_partner': fare_protobuf.conversion_partner,
            'tariff': MessageToDict(fare_protobuf.tariff),
            'popularity': fare_protobuf.popularity,
            'tariffs': self._tariffs_from_protobuf(fare_protobuf.tariffs),
            'promo': self._promo_from_protobuf(fare_protobuf.promo),
        }

    def _fares_from_protobuf(self, fares_field):
        return imap(self._fare_from_protobuf, fares_field)

    def _local_timestamp_from_protobuf(self, local_timestamp_protobuf):
        return {
            'local': local_timestamp_protobuf.local,
            'tzname': local_timestamp_protobuf.tzname,
            'offset': local_timestamp_protobuf.offset
        } if local_timestamp_protobuf.ByteSize() else None

    def _flight_from_protobuf(self, flight_protobuf):
        return {
            'arrival': self._local_timestamp_from_protobuf(flight_protobuf.arrival),
            'departure': self._local_timestamp_from_protobuf(flight_protobuf.departure),
            'to': flight_protobuf.to_id,
            'from': flight_protobuf.from_id,
            'company': flight_protobuf.company if flight_protobuf.HasField('company') else None,
            'aviaCompany': flight_protobuf.avia_company if flight_protobuf.HasField('avia_company') else None,
            'companyTariff': flight_protobuf.company_tariff if flight_protobuf.HasField('company_tariff') else None,
            'key': flight_protobuf.key,
            'number': flight_protobuf.number
        }

    def _flights_from_protobuf(self, protobuf_flights):
        flights_dict = dict()

        for k, v in protobuf_flights.iteritems():
            flights_dict[k] = self._flight_from_protobuf(v)

        return flights_dict

    def _polling_status_from_protobuf(self, protobuf_polling_status):
        return {
            'asked_partners': [partner for partner in protobuf_polling_status.asked_partners],
            'asked_partners_count': protobuf_polling_status.asked_partners_count,
            'remaining_partners': [partner for partner in protobuf_polling_status.remaining_partners],
            'remaining_partners_count': protobuf_polling_status.remaining_partners_count,
        }
