# -*- coding: utf-8 -*-
import logging
from datetime import datetime, timedelta
import enum
from typing import Any

from travel.avia.library.python.ticket_daemon.caches.services import get_service_by_code

logger = logging.getLogger(__name__)


class ErrorCodes(enum.OrderedDict):
    T_CODE = 't_code'
    INCORRECT_DATE_FORWARD = 'incorrect_date_forward'
    INCORRECT_DATE_BACKWARD = 'incorrect_date_backward'
    SERVICE = 'service'
    DATES_CONFLICT = 'dates_conflict'
    SAME_CITY = 'same_city'
    OTHER = 'other'


class ValidationError(ValueError):
    def __init__(self, description=None, error_code=None):
        super(ValueError, self).__init__(description)
        self.data = {ErrorCodes.OTHER: description}
        if error_code is not None:
            self.data = {error_code: description}


def validate_code(query):
    # type: (Any) -> None
    if query.t_code != 'plane':
        raise ValidationError('t_code should be "plane". Provided: %r' % query.t_code, error_code=ErrorCodes.T_CODE)


def validate_service(query):
    # type: (Any) -> None
    service = get_service_by_code(query.service)
    if not service:
        logger.warning(
            '%s query.service [%s] not in PROJECTS_GROUPS',
            query.qid_msg, query.service
        )
        raise ValidationError(
            'query.service [%s] not in PROJECTS_GROUPS' % query.service, error_code=ErrorCodes.SERVICE
        )


def validate_date(query):
    # type: (Any) -> None
    if query.date_forward < datetime.now().date():
        logger.warning('%s bad: date_forward in the past', query.qid_msg)
        raise ValidationError('date_forward in the past', error_code=ErrorCodes.INCORRECT_DATE_FORWARD)

    if query.date_forward > datetime.now().date() + timedelta(days=366):
        logger.warning('%s bad: date_forward in the far future', query.qid_msg)
        raise ValidationError('date_forward is too far in the future', error_code=ErrorCodes.INCORRECT_DATE_FORWARD)

    if query.date_backward and query.date_backward < query.date_forward:
        logger.warning('%s bad: date_backward is earlier than date_forward', query.qid_msg)
        raise ValidationError('date_backward is earlier than date_forward', error_code=ErrorCodes.DATES_CONFLICT)

    if query.date_backward and query.date_backward > datetime.now().date() + timedelta(days=366):
        logger.warning('%s bad: date_backward in the far future', query.qid_msg)
        raise ValidationError('date_backward is too far in the future', error_code=ErrorCodes.INCORRECT_DATE_BACKWARD)


def validate_custom_time(query):
    # type: (Any) -> None
    service = get_service_by_code(query.service)
    if query.meta['custom_store_time'] and not service.allow_custom_store_time:
        logger.warning('%s bad: custom_store_time not allowed for %s', query.service)
        raise ValidationError('custom_store_time not allowed', error_code=ErrorCodes.OTHER)


def validate_cities(query):
    # type: (Any) -> None
    city_from = query.city_from()
    city_to = query.city_to()
    if city_from and city_to and city_from.id == city_to.id:
        raise ValidationError('Same city from and city to', error_code=ErrorCodes.SAME_CITY)


def validate_s2s(query):
    # type: (Any) -> None
    if query.related_settlement_ids_from() & query.related_settlement_ids_to():
        raise ValidationError('s2s: Same city from and city to', error_code=ErrorCodes.SAME_CITY)


_VALIDATORS = (validate_code, validate_service, validate_date, validate_custom_time, validate_cities, validate_s2s)


def validate_query(query):
    # type: (Any) -> None
    for v in _VALIDATORS:
        v(query)
