# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from abc import abstractmethod
from collections import namedtuple
from typing import Optional, List, Dict, Any
import logging

from travel.avia.library.python.ticket_daemon import date as date_util

log = logging.getLogger(__name__)

BannedVariantParams = namedtuple('BannedVariantParams', 'tag')


class _DataManager(object):
    @abstractmethod
    def merge(self, old_variants_info, new_variant_info):
        # type: (Optional[List[Dict[Any]]], BannedVariantParams)->List[Dict[Any]]
        pass

    @property
    def ttl(self):
        # type: ()->int
        return 300


class Aeroflot(_DataManager):
    EXPIRES_AT = 'expires_at'

    def merge(self, old_variants_info, new_variant_info):
        # type: (Optional[Dict[Any]], BannedVariantParams)->Dict[Any]
        if not new_variant_info:
            return old_variants_info

        variants = {'tags': {}}
        if old_variants_info:
            variants = old_variants_info

        variants['tags'][new_variant_info.tag] = {Aeroflot.EXPIRES_AT: (date_util.unixtime() + self.ttl)}

        now = date_util.unixtime()
        for tag, values in variants['tags'].items():
            if values[Aeroflot.EXPIRES_AT] and values[Aeroflot.EXPIRES_AT] <= now:
                log.info(
                    'Tag %s expires at %s, now %s',
                    tag, values[Aeroflot.EXPIRES_AT], now,
                )
                del variants['tags'][tag]

        return variants


variants_updater_by_partner_code = {
    'aeroflot': Aeroflot,
}
