from typing import Union, Literal, Optional
from six.moves.urllib.parse import urlencode, urlunsplit, SplitResult

from travel.avia.library.python.urls.errors import MalformedUrlParameterValue
from travel.avia.library.python.urls.travel_base import TravelBase


class C:
    """Settlement code representation"""

    def __init__(self, id):
        # type: (int)->C
        self.id = id

    def __str__(self):
        return 'c{}'.format(self.id)

    def __repr__(self):
        return 'c{}'.format(self.id)


class S:
    """Station code representation"""

    def __init__(self, id):
        # type: (int)->S
        self.id = id

    def __str__(self):
        return 's{}'.format(self.id)

    def __repr__(self):
        return 's{}'.format(self.id)


class L:
    """Country code representation"""

    def __init__(self, id):
        # type: (int)->L
        self.id = id

    def __str__(self):
        return 'l{}'.format(self.id)

    def __repr__(self):
        return 'l{}'.format(self.id)


class TravelAviaSearch(TravelBase):
    URL_SCHEME = 'https'
    URL_PATH = '/avia/search/result/'

    def url(self, from_id, to_id, when='', return_date='', adults=1, children=0, infants=0, klass='economy'):
        # type: (Union[int, C, S, Literal['-me']], Union[int, C, S, Literal['-me']], str, Optional[str], int, int, int, str) -> str
        if not from_id:
            raise MalformedUrlParameterValue('from_id can not be null or empty')
        if not to_id:
            raise MalformedUrlParameterValue('to_id can not be null or empty')
        if not when:
            when = ''

        if isinstance(from_id, int):
            from_id = C(from_id)

        if isinstance(to_id, int):
            to_id = C(to_id)

        search_params = {
            'adult_seats': adults,
            'children_seats': children,
            'infant_seats': infants,
            'klass': klass,
            'fromId': str(from_id),
            'toId': str(to_id),
            'when': when,
        }
        if return_date:
            search_params['return_date'] = return_date

        return urlunsplit(
            SplitResult(
                scheme=self.URL_SCHEME,
                netloc=self.travel_host,
                path=self.URL_PATH,
                query=urlencode(search_params),
                fragment='',
            )
        )


class TravelAviaPrefilledFields(TravelBase):
    URL_SCHEME = 'https'
    URL_PATH = '/avia/'

    def url(self, from_id, to_id, when='', return_date=None, adults=1, children=0, infants=0, klass='economy'):
        # type: (Union[int, C, S, Literal['-me']], Union[int, C, S, Literal['-me']], str, Optional[str], int, int, int, str) -> str
        if not from_id:
            raise MalformedUrlParameterValue('from_id can not be null or empty')
        if not to_id:
            raise MalformedUrlParameterValue('to_id can not be null or empty')
        if not when:
            when = ''

        if isinstance(from_id, int):
            from_id = C(from_id)

        if isinstance(to_id, int):
            to_id = C(to_id)

        search_params = {
            'adult_seats': adults,
            'children_seats': children,
            'infant_seats': infants,
            'klass': klass,
            'fromId': str(from_id),
            'toId': str(to_id),
            'when': when,
        }
        if return_date is not None:
            search_params['return_date'] = return_date

        return urlunsplit(
            SplitResult(
                scheme=self.URL_SCHEME,
                netloc=self.travel_host,
                path=self.URL_PATH,
                query=urlencode(search_params),
                fragment='',
            )
        )
