# coding=utf-8

import re
import os.path
import logging
from urlparse import urljoin
from collections import defaultdict

import retrying
import requests
from yt.wrapper import TablePath

from travel.avia.library.python.lib_yt.client import configured_client

log = logging.getLogger(__name__)

SCHEMA = [
    {'type': 'string', 'name': 'fare_code'},
    {'type': 'int64', 'name': 'partner_id'},
    {'type': 'int64', 'name': 'airline_id'},
    {'type': 'boolean', 'name': 'company_in_reference'},
    {'type': 'boolean', 'name': 'segment_without_baggage', 'required': False},
    {'type': 'boolean', 'name': 'fare_without_baggage', 'required': False},
    {'type': 'int64', 'name': 'wrong_baggage_pieces', 'required': False},
    {'type': 'string', 'name': 'segment_baggage'},
]


def run_pipeline(ticket_daemon_api_host, yt_token, yt_proxy, variants_log_directory, date, output_table_path=None):
    log.info('Creating YT client')
    yt_client = configured_client(
        proxy=yt_proxy,
        token=yt_token,
    )
    fare_families_by_company_id = _get_fare_families_reference(ticket_daemon_api_host)

    output_table = build_table_with_unknown_fare_codes(
        yt_client,
        variants_log_directory,
        date,
        fare_families_by_company_id,
        output_table_path=output_table_path,
    )

    log.info('Output table: %s', output_table)


def _get_fare_families_reference(ticket_daemon_api_host):
    log.info('Getting fare families reference from %s', ticket_daemon_api_host)

    @retrying.retry(
        retry_on_result=lambda response: not response,
        stop_max_attempt_number=3,
        wait_fixed=1000,
    )
    def request_reference(url):
        try:
            log.info('Request: %s', url)
            response = requests.get(url)
            if response.status_code != 200:
                log.info('Unable to fetch from url: %s, result: %s', url, response.status_code)
                return None
            return response.json()['data']
        except Exception as e:
            log.info('Request error %r', e)
            return None

    path = 'jsendapi/fare-families'
    reference_url = urljoin(ticket_daemon_api_host, path)
    return request_reference(reference_url)


def build_table_with_unknown_fare_codes(yt_client, variants_log_directory, date, fare_families_by_company_id,
                                        output_table_path=None):
    """
    :param yt.wrapper yt_client:
    :param str variants_log_directory:
    :param datetime.date date:
    :param dict fare_families_by_company_id:
    :param Optional[string] output_table_path:
    """
    if output_table_path:
        if not yt_client.exists(output_table_path):
            yt_client.create('table', output_table_path, attributes={'schema': SCHEMA})
        output_table = output_table_path
    else:
        output_table = yt_client.create_temp_table(
            path='//home/avia/tmp',
            prefix='unknown_fare_codes_',
            expiration_timeout=12 * 60 * 60 * 1000,  # 12 hours
            attributes={'schema': SCHEMA},
        )
    log.info('Output table: %s', output_table)
    table_name = os.path.join(variants_log_directory, date.strftime('%Y-%m-%d'))
    table = TablePath(
        name=table_name,
        columns=['forward_segments', 'backward_segments', 'partner_id']
    )

    log.info('Run map reduce')
    yt_client.run_map_reduce(
        mapper=_get_variants_log_row_mapper(fare_families_by_company_id),
        reducer=_unknown_fare_codes_reducer,
        reduce_by=list(UnknownFareCodesLogRecord.__slots__),
        source_table=table,
        destination_table=output_table,
    )
    return output_table


def _unknown_fare_codes_reducer(key, records):
    yield key


def _get_variants_log_row_mapper(fare_families_by_company_id):
    def mapper(row):
        if row.get('forward_segments'):
            for segment in row.get('forward_segments'):
                for record in _segment_to_unknown_fare_codes_log_records(row, segment, fare_families_by_company_id):
                    yield record

        if row.get('backward_segments'):
            for segment in row.get('backward_segments'):
                for record in _segment_to_unknown_fare_codes_log_records(row, segment, fare_families_by_company_id):
                    yield record

    return mapper


def _segment_to_unknown_fare_codes_log_records(row, segment, fare_families_by_company_id):
    fare_code = segment.get('fare_code')
    if not fare_code:
        return
    airline_id = segment.get('airline_id')
    if not airline_id:
        return
    partner_id = row.get('partner_id')
    airline_id = str(airline_id)
    segment_baggage = segment.get('baggage')
    if airline_id not in fare_families_by_company_id:
        yield UnknownFareCodesLogRecord(partner_id, airline_id, fare_code, segment_baggage, company_in_reference=False).to_dict()
    else:
        for fare_family in fare_families_by_company_id.get(airline_id):
            pattern = fare_family.get('tariff_code_pattern')
            if re.match(pattern, fare_code):
                for record in _get_wrong_baggage_records(segment_baggage, fare_family, partner_id, airline_id, fare_code):
                    yield record
                return
        yield UnknownFareCodesLogRecord(partner_id, airline_id, fare_code, segment_baggage, company_in_reference=True).to_dict()


def _get_wrong_baggage_records(segment_baggage, fare_family, partner_id, airline_id, fare_code):
    included_in_segment, pieces_in_segment = _parse_baggage(segment_baggage)

    for term in fare_family['terms']:
        term_code = term.get('code')
        if term_code == 'baggage':
            for r in term['rules']:
                if r.get('ignore') or r.get('xpath') or r.get('external_xpath_ref'):
                    continue
                included_in_fare = True
                if included_in_fare != included_in_segment:
                    yield UnknownFareCodesLogRecord(
                        partner_id,
                        airline_id,
                        fare_code,
                        segment_baggage,
                        company_in_reference=True,
                        segment_without_baggage=True,
                    ).to_dict()
                    return
                pieces_in_fare = r.get('places')
                if pieces_in_fare != pieces_in_segment:
                    yield UnknownFareCodesLogRecord(
                        partner_id,
                        airline_id,
                        fare_code,
                        segment_baggage,
                        company_in_reference=True,
                        wrong_baggage_pieces=pieces_in_segment,
                    ).to_dict()
                return

    if included_in_segment:
        yield UnknownFareCodesLogRecord(
            partner_id,
            airline_id,
            fare_code,
            segment_baggage,
            company_in_reference=True,
            fare_without_baggage=True,
        ).to_dict()


def _parse_baggage(segment_baggage):
    included_in_segment = False
    pieces_in_segment = 0
    if segment_baggage:
        fields = ('included', 'pieces', 'weight')
        baggage = defaultdict(dict)
        key_parts = re.compile(r'((\d{0,2})([pdN]))')
        for (_, count, source), field in zip(key_parts.findall(segment_baggage), fields):
            baggage[field] = {
                'count': int(count),
                'source': 'partner' if source == 'p' else 'db'
            } if source != 'N' else None
        included_in_segment = baggage['included'].get('count') == 1 if baggage['included'] else False
        pieces_in_segment = baggage['pieces'].get('count', pieces_in_segment) if baggage['pieces'] else 0

    return included_in_segment, pieces_in_segment


class UnknownFareCodesLogRecord(object):
    __slots__ = (
        'partner_id',
        'airline_id',
        'fare_code',
        'segment_baggage',
        'company_in_reference',
        'segment_without_baggage',
        'wrong_baggage_pieces',
        'fare_without_baggage',
    )

    def __init__(
        self,
        partner_id,
        airline_id,
        fare_code,
        segment_baggage,
        company_in_reference=False,
        segment_without_baggage=None,
        wrong_baggage_pieces=None,
        fare_without_baggage=None,
    ):
        self.partner_id = partner_id
        self.airline_id = airline_id
        self.company_in_reference = company_in_reference
        self.fare_code = fare_code
        self.segment_baggage = segment_baggage
        self.segment_without_baggage = segment_without_baggage
        self.wrong_baggage_pieces = wrong_baggage_pieces
        self.fare_without_baggage = fare_without_baggage

    def to_dict(self):
        return dict(
            partner_id=self.partner_id,
            airline_id=int(self.airline_id),
            company_in_reference=self.company_in_reference,
            fare_code=self.fare_code,
            segment_without_baggage=self.segment_without_baggage,
            wrong_baggage_pieces=self.wrong_baggage_pieces,
            fare_without_baggage=self.fare_without_baggage,
            segment_baggage=self.segment_baggage,
        )
