import logging
import os
import socket
from typing import Sequence, Any

CONFIG_FILE = '/3proxy/3proxy.cfg'

SIRENA_ADDRESS = os.getenv('AVIA_SIRENA_HOST', '193.104.87.251')
STUDY_PORT = os.getenv('AVIA_SIRENA_STUDY_PORT', '34323')
TEST_PORT = os.getenv('AVIA_SIRENA_TEST_PORT', '34322')
PRODUCTION_PORT = os.getenv('AVIA_SIRENA_PROD_PORT', '34321')

AMADEUS_SFTP_REMOTE_HOST = os.getenv('AVIA_AMADEUS_SFTP_HOST', 'data-exchange.amadeus.com')
AMADEUS_SFTP_REMOTE_PORT = os.getenv('AVIA_AMADEUS_SFTP_REMOTE_PORT', '50522')
AMADEUS_SFTP_LOCAL_PORT = os.getenv('AVIA_AMADEUS_SFTP_LOCAL_PORT', '9022')


def _generate_services(service, num, port, internal_ip, external_ip):
    services = []
    for i in range(num):
        services.append('{service} -46 -p{port} {internal_ip} {external_ip}'.format(
            service=service,
            port=port,
            internal_ip=internal_ip,
            external_ip=external_ip,
        ))
    return '\n'.join(services)


def _generate_tcppm_service(
    internal_ip: str, external_ip: str, local_ports: Sequence[Any], remote_host: str, remote_ports: Sequence[Any],
):
    services = []
    if len(local_ports) != len(remote_ports):
        logging.error('TCP ports miscount')
    if not (local_ports and remote_ports):
        logging.error('Empty tcp ports list')
    internal_ip_opt = f'-i{internal_ip}' if internal_ip else ''
    external_ip_opt = f'-e{external_ip}' if external_ip else ''
    for local_port, remote_port in zip(local_ports, remote_ports):
        services.append('tcppm {}'.format(
            ' '.join(
                str(v) for v in [
                    internal_ip_opt, external_ip_opt, local_port, remote_host, remote_port,
                ] if v
            )
        ))
    return '\n'.join(services)


def main():
    s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    s.connect(('77.88.8.8', 80))
    ipv4 = s.getsockname()[0]
    s.close()
    ipv6 = os.getenv('DEPLOY_POD_IP_0_ADDRESS')

    with open(CONFIG_FILE, 'w') as f:
        f.write('# Generated config\n\n')
        f.write('# partners\n')
        f.write('auth strong\n')
        f.write('users {login}:CL:{password}\n'.format(
            login=os.getenv('AVIA_PROXY_LOGIN', 'avia'),
            password=os.getenv('AVIA_PROXY_PASSWORD', 'avia'),
        ))
        f.write('\nlog\n')
        f.write(_generate_services(
            service='proxy',
            num=int(os.getenv('AVIA_PROXY_SERVICE_THREADS', 1)),
            port=os.getenv('AVIA_HTTP_PROXY_PORT', '80'),
            internal_ip='-i{}'.format(ipv6) if ipv6 else '',
            external_ip='-e{}'.format(ipv4) if ipv4 else '',
        ))
        f.write('\n')
        f.write(_generate_services(
            service='ftppr',
            num=int(os.getenv('AVIA_FTP_PROXY_SERVICE_THREADS', 1)),
            port=os.getenv('AVIA_FTP_PROXY_PORT', '8021'),
            internal_ip='-i{}'.format(ipv6) if ipv6 else '',
            external_ip='-e{}'.format(ipv4) if ipv4 else '',
        ))
        f.write('\n\n')
        f.write('# amadeus\n')
        f.write('auth none\n')
        f.write(_generate_tcppm_service(
            internal_ip=ipv6,
            external_ip=ipv4,
            local_ports=(AMADEUS_SFTP_LOCAL_PORT,),
            remote_host=AMADEUS_SFTP_REMOTE_HOST,
            remote_ports=(AMADEUS_SFTP_REMOTE_PORT,),
        ))
        f.write('\n\n')
        f.write('# sirena\n')
        f.write(_generate_tcppm_service(
            internal_ip=ipv6,
            external_ip=ipv4,
            local_ports=(STUDY_PORT, TEST_PORT, PRODUCTION_PORT),
            remote_host=SIRENA_ADDRESS,
            remote_ports=(STUDY_PORT, TEST_PORT, PRODUCTION_PORT),
        ))
        f.write('\n')


if __name__ == '__main__':
    main()
