package main

import (
	"context"
	"fmt"
	"net"
	"net/http"
	"time"

	"github.com/go-chi/chi/v5"
	"github.com/jonboulle/clockwork"
	ghealth "google.golang.org/grpc/health"
	healthpb "google.golang.org/grpc/health/grpc_health_v1"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"a.yandex-team.ru/travel/avia/library/go/probes"
	"a.yandex-team.ru/travel/avia/personalization/internal/app/api"
	"a.yandex-team.ru/travel/avia/personalization/internal/caches/references"
	"a.yandex-team.ru/travel/avia/personalization/internal/lib/ydbutil"
	"a.yandex-team.ru/travel/avia/personalization/internal/services/dynamicresources"
	"a.yandex-team.ru/travel/avia/personalization/internal/services/personalsearch"
	grpchandlers "a.yandex-team.ru/travel/avia/personalization/internal/services/personalsearch/handlers/grpc"
	"a.yandex-team.ru/travel/avia/personalization/internal/tables"
	"a.yandex-team.ru/travel/library/go/configuration"
	grpcserver "a.yandex-team.ru/travel/library/go/grpcutil/server"
	"a.yandex-team.ru/travel/library/go/logging"
	"a.yandex-team.ru/travel/library/go/metrics"
	metricsserver "a.yandex-team.ru/travel/library/go/metrics/server"
	"a.yandex-team.ru/travel/library/go/syncutil"
	"a.yandex-team.ru/travel/library/go/tracing"
	tvmutil "a.yandex-team.ru/travel/library/go/tvm"
)

const serviceName = "personalization-api"

func main() {
	maxprocs.AdjustAuto()
	var wg syncutil.WaitGroup
	ctx, ctxCancel := context.WithCancel(context.Background())
	defer ctxCancel()

	// Config
	configLoader := configuration.NewDefaultConfitaLoader()
	if err := configLoader.Load(ctx, &api.Cfg); err != nil {
		panic(fmt.Sprintf("Failed to load confufuration: %v", err))
	}
	cfg := api.Cfg

	// Logging
	logger, err := logging.NewDeploy(&cfg.Logging)
	if err != nil {
		panic(fmt.Sprintf("Failed to create logger: %v", err))
	}

	tracerCloser := tracing.InitializeDefaultTracer(serviceName)
	defer func() {
		if err := tracerCloser.Close(); err != nil {
			logger.Error("Failed to close tracer", log.Error(err))
		}
		if err := logger.L.Sync(); err != nil {
			fmt.Println("Failed to close logger:", err)
		}
	}()

	// TVM
	var tvmClient tvm.Client
	tvmAllowedIds := tvmutil.TvmClientIDFromInt(cfg.Tvm.WhiteList)
	if cfg.Tvm.Enabled {
		tvmClient, err = tvmtool.NewDeployClient()
		if err != nil {
			logger.Fatal("Failed to create TVM client", log.Error(err))
		}
	}

	// YDB
	ydbConn, err := ydbutil.NewConnection(ctx, &cfg.Ydb)
	if err != nil {
		logger.Fatal("Failed to create YDB session pool", log.Error(err))
	}
	eventsTable := tables.NewUserEventsTable(logger, ydbConn, cfg.Ydb.Database, cfg.Tables.EventsTableName)
	if err := eventsTable.EnsureExists(ctx); err != nil {
		logger.Fatal("Failed to ensure user events table exists", log.Error(err))
	}
	logger.Info("YDB eventsTable initialized", log.Any("eventsTable", eventsTable))

	// Resources
	referencesRegistry, err := references.NewRegistry(cfg.Dicts, logger)
	if err != nil {
		logger.Fatal("Failed to create references registry", log.Error(err))
	}
	runDynamicResourcesService(cfg, logger, referencesRegistry)

	// Business logic
	clock := clockwork.NewRealClock()
	eventsProvider := personalsearch.NewEventsProvider(logger, eventsTable, referencesRegistry, clock)

	// gRPC health server
	healthServer := ghealth.NewServer()
	go func() {
	healthServerLoop:
		for range time.Tick(2 * time.Second) {
			select {
			case <-ctx.Done():
				healthServer.SetServingStatus(serviceName, healthpb.HealthCheckResponse_NOT_SERVING)
				break healthServerLoop
			default:
				healthServer.SetServingStatus(serviceName, healthpb.HealthCheckResponse_SERVING)
			}
		}
	}()

	// Metrics server
	rootRegistry := metrics.NewRegistryWithDeployTagsAndExplicitHost()
	appMetrics := metrics.NewAppMetrics(rootRegistry.WithPrefix("app"))
	metrics.SetGlobalAppMetrics(appMetrics)
	metrics.RunPerfMetricsUpdater(rootRegistry, metricsserver.DefaultMetricsConfig.PerfMetricsRefreshInterval)

	wg.Go(func() {
		logger.Info("Starting metrics server", log.String("address", metricsserver.DefaultMetricsConfig.Addr))
		err = metricsserver.RunMetricsHTTPServer(ctx, metricsserver.DefaultMetricsConfig, logger, rootRegistry)
		if err != nil {
			logger.Fatal("Metrics server error", log.Error(err))
		}
	})

	// gRPC server
	personalizationGRPCService := grpchandlers.NewPersonalizationService(eventsProvider, logger)
	grpcServiceRegisterers := []grpcserver.ServiceRegisterer{
		personalizationGRPCService.GetOldServiceRegisterer(),
		personalizationGRPCService.GetServiceRegisterer(),
	}

	grpcServer := grpcserver.NewDefaultGrpcServerBuilder(
		cfg.Grpc,
		grpcServiceRegisterers,
		logger,
		tvmClient,
		tvmAllowedIds,
		rootRegistry.WithPrefix("grpc"),
	).WithHealthServer(healthServer).Build()

	wg.Go(func() {
		logger.Info("Starting gRPC server", log.String("address", cfg.Grpc.Addr))
		if err := grpcServer.Run(ctx); err != nil {
			logger.Fatal("gRPC server error", log.Error(err))
		}
	})

	// Probes
	state := probes.NewState(logger, probes.OnStopAfter(ctxCancel))
	router := chi.NewRouter()
	probes.ChiBind(&cfg.Probes, router, state)

	// HTTP server
	httpServer := &http.Server{
		Addr:    cfg.HTTP.Addr,
		Handler: router,
	}

	wg.Go(func() {
		logger.Info("Starting HTTP server", log.String("address", cfg.HTTP.Addr))
		listener, err := net.Listen("tcp", cfg.HTTP.Addr)
		if err != nil {
			logger.Fatal("Failed to listen", log.Error(err), log.String("address", cfg.HTTP.Addr))
		}

		err = httpServer.Serve(listener)
		if !(ctx.Err() == context.Canceled && err == http.ErrServerClosed) {
			logger.Error("HTTP server error", log.Error(err))
		}
	})

	wg.Go(func() {
		doneChannel := ctx.Done()
		if doneChannel != nil {
			<-doneChannel
			logger.Info("Shutting down HTTP server", log.String("address", cfg.HTTP.Addr))
			if err := httpServer.Shutdown(context.Background()); err != nil {
				logger.Error("HTTP server error", log.Error(err))
			}
		}
	})

	wg.Wait()
}

func runDynamicResourcesService(cfg api.Config, logger log.Logger, dictsRegistry *references.Registry) {
	dynamicResourceService := dynamicresources.NewService(
		logger,
		cfg.DynamicResources,
		dictsRegistry.OnUpdate,
		func() error { return nil },
	)
	dynamicResourceService.BackroundRun()
}
