package main

import (
	"context"
	"fmt"
	"sync"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/travel/avia/personalization/internal/app/reader"
	"a.yandex-team.ru/travel/avia/personalization/internal/builders"
	"a.yandex-team.ru/travel/avia/personalization/internal/caches/references"
	"a.yandex-team.ru/travel/avia/personalization/internal/extractors"
	"a.yandex-team.ru/travel/avia/personalization/internal/lib/ydbutil"
	"a.yandex-team.ru/travel/avia/personalization/internal/repositories"
	"a.yandex-team.ru/travel/avia/personalization/internal/services/dynamicresources"
	"a.yandex-team.ru/travel/avia/personalization/internal/services/eventscollector"
	"a.yandex-team.ru/travel/avia/personalization/internal/services/profiling"
	"a.yandex-team.ru/travel/avia/personalization/internal/services/readiness"
	"a.yandex-team.ru/travel/avia/personalization/internal/tables"
	"a.yandex-team.ru/travel/library/go/configuration"
	"a.yandex-team.ru/travel/library/go/logging"
	"a.yandex-team.ru/travel/library/go/metrics"
	metricsserver "a.yandex-team.ru/travel/library/go/metrics/server"
)

func main() {
	// setting up infrastructure
	maxprocs.AdjustAuto()
	ctx, ctxCancel := context.WithCancel(context.Background())
	config := configuration.NewDefaultConfitaLoader()
	err := config.Load(ctx, &reader.Cfg)

	if err != nil {
		fmt.Println("can not load configuration:", err)
		ctxCancel()
		return
	}
	cfg := reader.Cfg
	logger, err := logging.NewDeploy(&cfg.Logging)
	if err != nil {
		fmt.Println("failed to create logger, err:", err)
		ctxCancel()
		return
	}

	defer func() {
		err = logger.L.Sync()
		if err != nil {
			fmt.Println("failed to close logger:", err)
		}
		ctxCancel()
	}()

	rootRegistry := metrics.NewRegistryWithDeployTagsAndExplicitHost()
	appMetrics := metrics.NewAppMetrics(rootRegistry.WithPrefix("app"))
	metrics.SetGlobalAppMetrics(appMetrics)
	metrics.RunPerfMetricsUpdater(rootRegistry, metricsserver.DefaultMetricsConfig.PerfMetricsRefreshInterval)

	ydbConn, err := ydbutil.NewConnection(ctx, &cfg.Ydb)
	if err != nil {
		logger.Error("failed to create YDB session pool", log.Error(err))
		return
	}
	eventsTable := tables.NewUserEventsTable(logger, ydbConn, cfg.Ydb.Database, cfg.Tables.EventsTableName)
	if err := eventsTable.EnsureExists(ctx); err != nil {
		logger.Fatal("failed to ensure user events table exists", log.Error(err))
	}

	referencesRegistry, err := references.NewRegistry(cfg.Dicts, logger)
	if err != nil {
		logger.Error("failed to create references registry", log.Error(err))
		return
	}

	logger.Info("configuring hotels provider")
	hotelsProvider := repositories.NewHotelsProvider(
		logger,
		&cfg.HotelsReference,
	)
	err = hotelsProvider.Precache()
	if err != nil {
		logger.Error("failed to precache hotels reference", log.Error(err))
		return
	}

	runDynamicResourcesService(cfg, logger, referencesRegistry, hotelsProvider)

	go func() {
		err = metricsserver.RunMetricsHTTPServer(context.Background(), metricsserver.DefaultMetricsConfig, logger, rootRegistry)
		if err != nil {
			logger.Fatal("Error while starting metrics server", log.Error(err))
		}
	}()

	settlementExtractor := extractors.NewSettlementExtractor(
		extractors.NewStationIDToSettlementIDMapper(referencesRegistry),
		referencesRegistry,
	)
	eventBuilder := builders.NewEventBuilder(referencesRegistry, hotelsProvider, settlementExtractor)
	eventsCollector := eventscollector.NewService(cfg.Logbroker, cfg.EventsCollector, logger, eventsTable, eventBuilder)

	runReadinessServer(cfg, eventsCollector)
	runProfilingService(cfg)
	wg := sync.WaitGroup{}
	wg.Add(1)

	go func() {
		defer wg.Done()
		eventsCollector.Run(ctx)
	}()

	logger.Info("Started")
	if !eventsCollector.HasEnabledSources() {
		logger.Info("all collectors are disabled so reader will just idle")
		wg.Add(1)
	}
	wg.Wait()
}

func runProfilingService(cfg reader.Config) {
	profilingService := profiling.NewService(cfg.Profiling)
	profilingService.BackroundRun()
}

func runDynamicResourcesService(
	cfg reader.Config,
	logger log.Logger,
	dictsRegistry *references.Registry,
	hotelsProvider *repositories.HotelsProvider,
) {
	dynamicResourceService := dynamicresources.NewService(
		logger,
		cfg.DynamicResources,
		dictsRegistry.OnUpdate,
		hotelsProvider.Precache,
	)
	dynamicResourceService.BackroundRun()
}

func runReadinessServer(cfg reader.Config, eventsCollector *eventscollector.Service) {
	readinessService := readiness.NewService(
		cfg.Readiness,
		eventsCollector.IsReady,
	)
	readinessService.BackroundRun()
}
