package builders

import (
	"fmt"
	"strconv"
	"strings"
	"time"

	timeformats "cuelang.org/go/pkg/time"

	portalproto "a.yandex-team.ru/travel/api/proto/hotels_portal"
	"a.yandex-team.ru/travel/avia/personalization/internal/caches/references"
	"a.yandex-team.ru/travel/avia/personalization/internal/events"
	"a.yandex-team.ru/travel/avia/personalization/internal/lib"
	"a.yandex-team.ru/travel/avia/personalization/internal/repositories"
	"a.yandex-team.ru/travel/avia/personalization/internal/ytlogs"
	"a.yandex-team.ru/travel/proto/cpa"
)

type SettlementExtractor interface {
	GetSettlementIDByCode(string) (int32, error)
	GetSettlementIDByPointKey(string) (int32, error)
}

const AviaWizardServiceClass = "economy"

type EventBuilder struct {
	references          *references.Registry
	hotelsProvider      *repositories.HotelsProvider
	settlementExtractor SettlementExtractor
}

func NewEventBuilder(
	references *references.Registry,
	hotelsProvider *repositories.HotelsProvider,
	settlementExtractor SettlementExtractor,
) *EventBuilder {
	return &EventBuilder{references: references, hotelsProvider: hotelsProvider, settlementExtractor: settlementExtractor}
}

func (e *EventBuilder) FromAviaTravelSearch(record ytlogs.AviaUserSearchRecord) (events.AviaUserSearch, error) {
	fromID, err := e.settlementExtractor.GetSettlementIDByPointKey(record.FromID)
	if err != nil {
		return events.AviaUserSearch{}, err
	}
	toID, err := e.settlementExtractor.GetSettlementIDByPointKey(record.ToID)
	if err != nil {
		return events.AviaUserSearch{}, err
	}
	return events.AviaUserSearch{
		Unixtime:         record.Unixtime,
		SettlementFromID: fromID,
		SettlementToID:   toID,
		AdultSeats:       record.AdultSeats,
		ChildrenSeats:    record.ChildrenSeats,
		InfantSeats:      record.InfantSeats,
		When:             record.When,
		ReturnDate:       record.ReturnDate,
		NationalVersion:  record.NationalVersion,
		Lang:             record.Lang,
		Klass:            record.Klass,
		YandexUID:        record.YandexUID,
		PassportID:       record.PassportID,
	}, nil
}

func (e *EventBuilder) FromWizardQuery(record ytlogs.AviaWizardQueryRecord) (events.AviaUserSearch, error) {
	event := events.AviaUserSearch{
		Unixtime:         record.Unixtime,
		SettlementFromID: int32(lib.PointKeyToID(record.FromID)),
		SettlementToID:   int32(lib.PointKeyToID(record.ToID)),
		When:             record.When,
		ReturnDate:       record.ReturnDate,
		NationalVersion:  record.NationalVersion,
		Lang:             record.Lang,
		Klass:            AviaWizardServiceClass,
		YandexUID:        record.YandexUID,
		PassportID:       record.PassportID,
	}
	passengers := strings.Split(record.PassengersInfo, ",")
	event.AdultSeats = parseSeats(passengers[0])
	event.ChildrenSeats = parseSeats(passengers[1])
	event.InfantSeats = parseSeats(passengers[2])

	return event, nil
}

func (e *EventBuilder) FromTravelOfferCache(record ytlogs.TravelHotelsOffercacheRecord) (events.HotelsUserSearch, error) {
	event := events.HotelsUserSearch{
		Unixtime:     record.Unixtime,
		CheckInDate:  record.Resp.CheckInDate,
		CheckOutDate: record.Resp.CheckOutDate,
		Duration:     record.Resp.Duration,
		Ages:         record.Resp.Ages,
		YandexUID:    record.Req.YandexUID,
		PassportID:   record.Req.PassportUID,
	}

	passengers, err := e.parseAges(event.Ages)
	if err != nil {
		return event, err
	}
	event.AdultSeats = passengers.adult
	event.ChildrenSeats = passengers.children
	event.InfantSeats = passengers.infant

	for permalink := range record.Resp.Hotels {
		permalink, _ := strconv.Atoi(permalink)
		hotel := e.hotelsProvider.GetHotelByPermalink(permalink)
		if hotel != nil {
			settlement, found := e.references.Settlements.GetByGeoID(int32(hotel.GeoID))
			if !found {
				continue
			}

			event.SettlementToID = settlement.GetId()
			break
		}
	}

	return event, nil
}

func (e *EventBuilder) FromTravelHotelsSearch(record *portalproto.THotelsSearchLogRecord) (events.HotelsUserSearch, error) {
	offerCacheParams := record.Response.GetOfferCacheParams()
	if offerCacheParams == nil {
		return events.HotelsUserSearch{}, fmt.Errorf("unable to extract offerCacheParams")
	}

	event := events.HotelsUserSearch{
		Unixtime:     uint32(record.GetRequestTimestamp()),
		CheckInDate:  offerCacheParams.GetCheckInDate(),
		CheckOutDate: offerCacheParams.GetCheckOutDate(),
		Ages:         offerCacheParams.GetAges(),
		YandexUID:    record.RequestHeaders.GetYandexUid(),
		PassportID:   record.RequestHeaders.GetPassportUid(),
	}

	passengers, err := e.parseAges(event.Ages)
	if err != nil {
		return event, err
	}
	event.AdultSeats = passengers.adult
	event.ChildrenSeats = passengers.children
	event.InfantSeats = passengers.infant

	geoID := int(record.Request.GetGeoId())
	if geoID == 0 && record.Request.GetPermalink() != "" {
		permalink, _ := strconv.Atoi(record.Request.GetPermalink())
		hotel := e.hotelsProvider.GetHotelByPermalink(permalink)
		if hotel == nil {
			return events.HotelsUserSearch{}, nil
		}
		geoID = hotel.GeoID
	}

	if settlement, found := e.references.Settlements.GetByGeoID(int32(geoID)); found {
		event.SettlementToID = settlement.GetId()
	}
	return event, nil
}

type passengers struct {
	adult    uint8
	children uint8
	infant   uint8
}

func (e *EventBuilder) parseAges(ages string) (*passengers, error) {
	travellers := new(passengers)
	for _, value := range strings.Split(ages, ",") {
		intValue, err := strconv.Atoi(value)
		if err != nil {
			return nil, err
		}
		if intValue < 2 {
			travellers.infant++
		} else if intValue >= 12 {
			travellers.adult++
		} else {
			travellers.children++
		}
	}

	return travellers, nil
}

func (e *EventBuilder) FromAviaCPAOrder(record *cpa.TGenericOrderInfo, timestamp uint64) (events.AviaCPAOrder, error) {
	orderItem := record.GetOrderItems()[0]
	aviaOrder := orderItem.GetOrderInfoAvia()
	aviaLabel := record.GetLabel().GetLabelAvia()
	fromID, err := e.settlementExtractor.GetSettlementIDByCode(aviaOrder.Origin)
	if err != nil {
		return events.AviaCPAOrder{}, fmt.Errorf("failed to extract FromID from origin %s", aviaOrder.Origin)
	}
	toID, err := e.settlementExtractor.GetSettlementIDByCode(aviaOrder.Destination)
	if err != nil {
		return events.AviaCPAOrder{}, fmt.Errorf("failed to extract ToID from destination %s", aviaOrder.Destination)
	}
	return events.AviaCPAOrder{
		Unixtime:         uint32(timestamp),
		DateForward:      aviaOrder.GetDateForward(),
		DateBackward:     aviaOrder.GetDateBackward(),
		SettlementFromID: fromID,
		SettlementToID:   toID,
		Status:           orderItem.GetOrderInfoAvia().OrderStatus.String(),
		AdultSeats:       uint8(aviaLabel.AdultSeats),
		ChildrenSeats:    uint8(aviaLabel.ChildrenSeats),
		InfantSeats:      uint8(aviaLabel.InfantSeats),
		ServiceClass:     aviaLabel.ServiceClass,
		OrderID:          e.buildOrderID(aviaOrder.GetPartnerName(), aviaOrder.GetPartnerOrderId()),
		YandexUID:        aviaLabel.GetYandexUid(),
		PassportID:       aviaLabel.GetPassportId(),
	}, nil
}

func (e *EventBuilder) FromHotelsCPAOrder(record *cpa.TGenericOrderInfo, timestamp uint64) (events.HotelsCPAOrder, error) {
	orderItem := record.GetOrderItems()[0]
	hotelsLabel := record.GetLabel().GetLabelHotels()
	hotelsOrder := orderItem.GetOrderInfoHotels()
	event := events.HotelsCPAOrder{
		Unixtime:     uint32(timestamp),
		CheckInDate:  orderItem.GetOrderInfoHotels().GetCheckIn(),
		CheckOutDate: orderItem.GetOrderInfoHotels().GetCheckOut(),
		Duration:     e.getDuration(orderItem.GetOrderInfoHotels().GetCheckIn(), orderItem.GetOrderInfoHotels().GetCheckOut()),
		Status:       orderItem.GetOrderInfoHotels().OrderStatus.String(),
		OrderID:      e.buildOrderID(hotelsOrder.GetPartnerName(), hotelsOrder.GetPartnerOrderId()),
		YandexUID:    hotelsLabel.GetYandexUid(),
		PassportID:   hotelsLabel.GetPassportUid(),
	}
	permalink := record.GetLabel().GetLabelHotels().GetPermalink()
	hotel := e.hotelsProvider.GetHotelByPermalink(int(permalink))
	if hotel != nil {
		settlement, found := e.references.Settlements.GetByGeoID(int32(hotel.GeoID))
		if !found {
			return event, nil
		}
		event.SettlementToID = settlement.GetId()
	}
	return event, nil
}

func (e *EventBuilder) buildOrderID(partnerName string, partnerOrderID string) string {
	return fmt.Sprintf("%s_%s", partnerName, partnerOrderID)
}

func (e *EventBuilder) getDuration(rawCheckInDate, rawCheckOutDate string) int {
	checkInDate, _ := time.Parse(timeformats.RFC3339Date, rawCheckInDate)
	checkOutDate, _ := time.Parse(timeformats.RFC3339Date, rawCheckOutDate)
	if checkInDate.IsZero() || checkOutDate.IsZero() {
		return 0
	}
	return int(checkOutDate.Sub(checkInDate).Hours()) / 24
}

func parseSeats(passengers string) uint8 {
	seats, _ := strconv.ParseUint(passengers, 10, 8)
	return uint8(seats)
}
