package references

import (
	"fmt"
	"io"
	"os"
	"path/filepath"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/library/go/dicts/base"
)

type Config struct {
	ResourcesPath       string `config:"DICTS_RESOURCES_PATH" yaml:"resources_path"`
	UseDynamicResources bool   `yaml:"use_dynamic_resources"`
}

var DefaultConfig = Config{
	ResourcesPath:       "/app/data",
	UseDynamicResources: false,
}

type Registry struct {
	writerByPath         map[string]UpdatableWriter
	Settlements          *SettlementsRepository
	StationCodes         *StationCodesRepository
	Stations             *StationsRepository
	StationToSettlements *StationToSettlementsRepository
	config               Config
}

type RepositoryUpdater interface {
	Populate(writer io.Writer) error
}

type UpdatableWriter interface {
	io.Writer
	UpdateFromSource(RepositoryUpdater) error
}

func populateMany(path2writer map[string]UpdatableWriter) error {
	for path, writer := range path2writer {
		if err := populateFromPath(path, writer); err != nil {
			return err
		}
	}
	return nil
}

func populateFromPath(path string, writer UpdatableWriter) error {
	file, err := os.Open(path)
	if err != nil {
		return fmt.Errorf("pkg.dicts.registry:populateFromPath %w", err)
	}
	defer file.Close()
	bytesIterator, _ := base.BuildIteratorFromReader(file)
	err = bytesIterator.Populate(writer)
	if err != nil {
		return fmt.Errorf("pkg.dicts.registry:populateFromPath %w", err)
	}
	return nil
}

func NewRegistry(c Config, logger log.Logger) (*Registry, error) {
	r := &Registry{
		Settlements:          NewSettlementsRepository(),
		StationToSettlements: NewStationToSettlementsRepository(),
		Stations:             NewStationsRepository(),
		StationCodes:         NewStationCodesRepository(),
		config:               c,
	}

	resourcePath, err := getResourcePath(c)
	if err != nil {
		return nil, fmt.Errorf("pkg.dicts.registry:NewRegistry %w", err)
	}

	r.updateWriterByPath(resourcePath)
	err = populateMany(r.writerByPath)
	if err != nil {
		return nil, fmt.Errorf("pkg.dicts.registry:NewRegistry %w", err)
	}
	return r, nil
}

func (r *Registry) OnUpdate() error {
	resourcePath, err := getResourcePath(r.config)
	if err != nil {
		return fmt.Errorf("pkg.dicts.registry:OnUpdate %w", err)
	}
	r.updateWriterByPath(resourcePath)

	for path, writer := range r.writerByPath {
		if err := r.updateFromPath(path, writer); err != nil {
			return err
		}
	}
	return nil
}

func (r *Registry) updateFromPath(path string, writer UpdatableWriter) error {
	file, err := os.Open(path)
	if err != nil {
		return fmt.Errorf("pkg.dicts.registry:updateFromPath %w", err)
	}
	defer file.Close()
	bytesIterator, _ := base.BuildIteratorFromReader(file)
	if err := writer.UpdateFromSource(bytesIterator); err != nil {
		return fmt.Errorf("pkg.dicts.registry:updateFromPath %w", err)
	}
	return nil
}

func (r *Registry) updateWriterByPath(resourcePath string) {
	r.writerByPath = map[string]UpdatableWriter{
		resourcePath + "/settlement.data":         r.Settlements,
		resourcePath + "/station_code.data":       r.StationCodes,
		resourcePath + "/station.data":            r.Stations,
		resourcePath + "/settlement2station.data": r.StationToSettlements,
	}
}

func getResourcePath(config Config) (string, error) {
	resourcePath := config.ResourcesPath
	if config.UseDynamicResources {
		symlinkTarget, err := filepath.EvalSymlinks(config.ResourcesPath)
		if err != nil {
			return "", err
		}
		resourcePath = filepath.Dir(symlinkTarget)
	}
	return resourcePath, nil
}
