package references

import (
	"strings"

	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/travel/library/go/dicts/base"
	"a.yandex-team.ru/travel/library/go/dicts/extensions"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
)

type SettlementsRepository struct {
	*base.Repository
	*extensions.SettlementByID
	*extensions.SettlementByGeoID
	*SettlementByCodeExtension
	settlementByIATA   map[string]*rasp.TSettlement
	settlementBySirena map[string]*rasp.TSettlement
}

func NewSettlementsRepository() *SettlementsRepository {
	newProto := func() proto.Message {
		return &rasp.TSettlement{}
	}

	repo := &SettlementsRepository{
		Repository:                base.NewRepository("Settlement", newProto),
		SettlementByID:            extensions.NewSettlementByID(),
		SettlementByGeoID:         extensions.NewSettlementByGeoID(),
		SettlementByCodeExtension: NewSettlementByCodeExtension(),
	}
	repo.AddExtensions(
		repo.SettlementByID,
		repo.SettlementByGeoID,
		repo.SettlementByCodeExtension,
	)
	return repo
}

func (cr *SettlementsRepository) UpdateFromSource(iterator RepositoryUpdater) error {
	newState := NewSettlementsRepository()
	if err := iterator.Populate(newState); err != nil {
		return err
	}
	*cr = *newState
	return nil
}

type SettlementByCodeExtension struct {
	settlementByIATA   map[string]*rasp.TSettlement
	settlementBySirena map[string]*rasp.TSettlement
}

func NewSettlementByCodeExtension() *SettlementByCodeExtension {
	return &SettlementByCodeExtension{
		settlementByIATA:   make(map[string]*rasp.TSettlement),
		settlementBySirena: make(map[string]*rasp.TSettlement),
	}
}

func (ext *SettlementByCodeExtension) AddProto(message proto.Message) {
	settlement := message.(*rasp.TSettlement)
	if settlement.GetIata() != "" {
		ext.settlementByIATA[strings.ToUpper(settlement.Iata)] = settlement
	}
	if settlement.GetSirenaId() != "" {
		ext.settlementByIATA[strings.ToUpper(settlement.SirenaId)] = settlement
	}
}

func (ext *SettlementByCodeExtension) GetByCode(code string) (*rasp.TSettlement, bool) {
	if code == "" {
		return nil, false
	}
	code = strings.ToUpper(code)
	if result, found := ext.settlementByIATA[code]; found {
		return result, found
	}
	result, found := ext.settlementBySirena[code]
	return result, found
}
