package references

import (
	"fmt"

	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/travel/library/go/dicts/repository"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
)

type StationsRepository struct {
	baseRepository       *repository.StationRepository
	stationsBySettlement map[int][]*rasp.TStation
}

func NewStationsRepository() *StationsRepository {
	return &StationsRepository{
		baseRepository:       repository.NewStationRepository(),
		stationsBySettlement: make(map[int][]*rasp.TStation),
	}
}

func (cr *StationsRepository) Write(b []byte) (int, error) {
	station := &rasp.TStation{}
	if err := proto.Unmarshal(b, station); err != nil {
		return 0, fmt.Errorf("StationRepository:Write: %w", err)
	}
	cr.baseRepository.Add(station)
	if station.SettlementId != 0 {
		cr.stationsBySettlement[int(station.SettlementId)] = append(cr.stationsBySettlement[int(station.SettlementId)], station)
	}
	return len(b), nil
}

func (cr *StationsRepository) Get(id int) (*rasp.TStation, bool) {
	station, found := cr.baseRepository.Get(id)
	return station, found
}

func (cr *StationsRepository) GetBySettlementID(id int) ([]*rasp.TStation, bool) {
	result, found := cr.stationsBySettlement[id]
	return result, found
}

func (cr *StationsRepository) UpdateFromSource(iterator RepositoryUpdater) error {
	newState := NewStationsRepository()
	if err := iterator.Populate(newState); err != nil {
		return err
	}
	*cr = *newState
	return nil
}
