package collectors

import (
	"context"
	"encoding/json"
	"fmt"
	"time"

	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/personalization/internal/auth"
	"a.yandex-team.ru/travel/avia/personalization/internal/consts"
	"a.yandex-team.ru/travel/avia/personalization/internal/events"
	"a.yandex-team.ru/travel/avia/personalization/internal/extractors"
	"a.yandex-team.ru/travel/avia/personalization/internal/metrics"
	"a.yandex-team.ru/travel/avia/personalization/internal/tables"
	"a.yandex-team.ru/travel/avia/personalization/internal/ytlogs"
)

const (
	aviaUserSearchLogMetricName        = "avia-user-search-log"
	aviaUserSearchLogTimingsMetricName = "avia-user-search-log.timings"
)

type AviaUserSearchEventBuilder interface {
	FromAviaTravelSearch(record ytlogs.AviaUserSearchRecord) (events.AviaUserSearch, error)
}

type AviaUserSearchLogCollector struct {
	logger       log.Logger
	eventsTable  *tables.UserEventsTable
	eventBuilder AviaUserSearchEventBuilder
}

func NewAviaUserSearchLogCollector(
	logger log.Logger,
	eventsTable *tables.UserEventsTable,
	eventBuilder AviaUserSearchEventBuilder,
) *AviaUserSearchLogCollector {
	return &AviaUserSearchLogCollector{
		logger:       logger.WithName("AviaUserSearchLogCollector"),
		eventsTable:  eventsTable,
		eventBuilder: eventBuilder,
	}
}

func (c *AviaUserSearchLogCollector) OnMessage(message []byte) {
	startTime := time.Now()
	defer metrics.WriteTimings(aviaUserSearchLogTimingsMetricName, startTime, map[string]string{"type": "all"})
	ctx := context.Background()

	parsedMessage, err := c.ParseMessage(message)
	if err != nil {
		c.logger.Error(
			"Cannot read or handle status message",
			log.Error(err),
			log.Any("message", message),
		)
		return
	}

	c.logger.Debug(
		"Parsed Message",
		log.Any("message", parsedMessage),
	)
	if parsedMessage.YandexUID == "" && parsedMessage.PassportID == "" {
		c.logger.Debug("Skip message without YandexUID and PassportID")
		return
	}
	event, err := c.eventBuilder.FromAviaTravelSearch(parsedMessage)
	if err != nil {
		if _, isErrStationWithoutSettlement := err.(*extractors.ErrStationWithoutSettlement); isErrStationWithoutSettlement {
			c.logger.Warn("Cannot prepare DB event", log.Error(err))
		} else {
			c.logger.Error("Cannot prepare DB event", log.Error(err))
		}
		return
	}

	serializedEvent, err := json.Marshal(event)
	if err != nil {
		c.logger.Error("Cannot marshal AviaUserSearchLog message", log.Any("message", message))
	}
	defer metrics.WriteTimings(aviaUserSearchLogTimingsMetricName, startTime, map[string]string{"type": "to_be_saved"})
	authType := auth.TypeYandexUID
	authValue := event.YandexUID
	if event.PassportID != "" {
		authType = auth.TypePassportID
		authValue = event.PassportID
	}
	eventsTableEntry := tables.UserEventEntry{
		AuthType:  authType,
		AuthValue: authValue,
		Service:   consts.AviaServiceID,
		EventType: consts.EventTypeSearch,
		EventKey:  fmt.Sprintf("c%d_c%d", event.SettlementFromID, event.SettlementToID),
		EventData: string(serializedEvent),
		CreatedAt: event.Unixtime,
		ExpiresAt: event.Unixtime + uint32(consts.EventTTL.Milliseconds()),
	}
	ctx = ctxlog.WithFields(
		ctx,
		log.UInt8("service", consts.AviaServiceID),
		log.UInt8("eventType", consts.EventTypeSearch),
		log.UInt8("authType", eventsTableEntry.AuthType),
		log.String("authValue", eventsTableEntry.AuthValue),
		log.UInt32("createdAt", eventsTableEntry.CreatedAt),
	)
	err = c.eventsTable.Upsert(ctx, eventsTableEntry)
	if err != nil {
		ctxlog.Error(ctx, c.logger, "Cannot save message to YDB", log.Error(err))
		metrics.IncCounterMetric(aviaUserSearchLogMetricName, map[string]string{"type": "error"})
		return
	}
	ctxlog.Info(ctx, c.logger, "UserSearch event saved")
	metrics.IncCounterMetric(aviaUserSearchLogMetricName, map[string]string{"type": "success"})
}

func (c *AviaUserSearchLogCollector) OnError(decompressedMessage persqueue.ReadMessageOrError) {
	metrics.IncCounterMetric(aviaUserSearchLogMetricName, map[string]string{"type": "error"})
}

func (c *AviaUserSearchLogCollector) ParseMessage(message []byte) (ytlogs.AviaUserSearchRecord, error) {
	var record ytlogs.AviaUserSearchRecord

	err := json.Unmarshal(message, &record)
	if err != nil {
		return record, xerrors.Errorf(
			"cannot unmarshal userSearches: %s: %w",
			string(message),
			err,
		)
	}

	return record, nil
}
