package collectors

import (
	"context"
	"encoding/json"
	"fmt"
	"time"

	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/personalization/internal/auth"
	"a.yandex-team.ru/travel/avia/personalization/internal/consts"
	"a.yandex-team.ru/travel/avia/personalization/internal/events"
	"a.yandex-team.ru/travel/avia/personalization/internal/extractors"
	"a.yandex-team.ru/travel/avia/personalization/internal/metrics"
	"a.yandex-team.ru/travel/avia/personalization/internal/tables"
	"a.yandex-team.ru/travel/avia/personalization/internal/ytlogs"
)

const (
	aviaWizardQueryLogMetricName        = "avia-wizard-query-log"
	aviaWizardQueryLogTimingsMetricName = "avia-wizard-query-log.timings"
)

type AviaWizardSearchEventBuilder interface {
	FromWizardQuery(record ytlogs.AviaWizardQueryRecord) (events.AviaUserSearch, error)
}

type AviaWizardQueryLogCollector struct {
	logger       log.Logger
	eventsTable  *tables.UserEventsTable
	eventBuilder AviaWizardSearchEventBuilder
}

func NewAviaWizardQueryLogCollector(
	logger log.Logger,
	eventsTable *tables.UserEventsTable,
	eventBuilder AviaWizardSearchEventBuilder,
) *AviaWizardQueryLogCollector {
	return &AviaWizardQueryLogCollector{
		logger:       logger.WithName("AviaWizardQueryLogCollector"),
		eventsTable:  eventsTable,
		eventBuilder: eventBuilder,
	}
}

func (c *AviaWizardQueryLogCollector) OnMessage(message []byte) {
	startTime := time.Now()
	metrics.WriteTimings(aviaWizardQueryLogTimingsMetricName, startTime, map[string]string{"type": "all"})
	ctx := context.Background()

	parsedMessage, err := c.ParseMessage(message)
	if err != nil {
		c.logger.Error(
			"Cannot read or handle status message",
			log.Error(err),
			log.Any("message", message),
		)
		return
	}

	c.logger.Debug(
		"Parsed Message",
		log.Any("message", parsedMessage),
	)
	if !parsedMessage.AviaDynamic || parsedMessage.Organic {
		c.logger.Debug("Skip non dynamic query message")
		return
	}
	if len(parsedMessage.RawArgs.AviaDynamic) == 0 || parsedMessage.RawArgs.AviaDynamic[0].FirstRequest != 1 {
		c.logger.Debug("Skip polling query message")
		return
	}
	if parsedMessage.YandexUID == "" && parsedMessage.PassportID == "" {
		c.logger.Debug("Skip message without YandexUID and PassportID")
		return
	}
	event, err := c.eventBuilder.FromWizardQuery(parsedMessage)
	if err != nil {
		if _, isErrStationWithoutSettlement := err.(*extractors.ErrStationWithoutSettlement); isErrStationWithoutSettlement {
			c.logger.Error("Cannot prepare DB event", log.Error(err))
		}
		return
	}

	serializedEvent, err := json.Marshal(event)
	if err != nil {
		c.logger.Error("Cannot marshal AviaWizardQueryLog message", log.Any("message", message))
		return
	}
	defer metrics.WriteTimings(aviaWizardQueryLogTimingsMetricName, startTime, map[string]string{"type": "to_be_saved"})
	eventStr := string(serializedEvent)
	authType := auth.TypeYandexUID
	authValue := event.YandexUID
	if event.PassportID != "" {
		authType = auth.TypePassportID
		authValue = event.PassportID
	}
	userEventsTableEntry := tables.UserEventEntry{
		AuthType:  authType,
		AuthValue: authValue,
		Service:   consts.AviaServiceID,
		EventType: consts.EventTypeSearch,
		EventKey:  c.buildEventKey(event),
		EventData: eventStr,
		CreatedAt: parsedMessage.Unixtime,
		ExpiresAt: parsedMessage.Unixtime + uint32(consts.EventTTL.Milliseconds()),
	}
	err = c.eventsTable.Upsert(ctx, userEventsTableEntry)
	if err != nil {
		c.logger.Error(
			"Cannot save message to YDB", log.Error(err),
			log.UInt8("service", userEventsTableEntry.Service),
			log.UInt8("eventType", userEventsTableEntry.EventType),
			log.UInt8("authType", userEventsTableEntry.AuthType),
			log.String("authValue", userEventsTableEntry.AuthValue),
			log.String("eventKey", userEventsTableEntry.EventKey),
			log.UInt32("createdAt", userEventsTableEntry.CreatedAt),
		)
		metrics.IncCounterMetric(aviaWizardQueryLogMetricName, map[string]string{"type": "error"})
		return
	}
	c.logger.Info(
		"WizardSearch Event saved",
		log.UInt8("service", userEventsTableEntry.Service),
		log.UInt8("eventType", userEventsTableEntry.EventType),
		log.UInt8("authType", userEventsTableEntry.AuthType),
		log.String("authValue", userEventsTableEntry.AuthValue),
		log.String("eventKey", userEventsTableEntry.EventKey),
		log.UInt32("createdAt", userEventsTableEntry.CreatedAt),
	)
	metrics.IncCounterMetric(aviaWizardQueryLogMetricName, map[string]string{"type": "success"})
}

func (c *AviaWizardQueryLogCollector) OnError(decompressedMessage persqueue.ReadMessageOrError) {
	metrics.IncCounterMetric(aviaWizardQueryLogMetricName, map[string]string{"type": "error"})
}

func (c *AviaWizardQueryLogCollector) ParseMessage(message []byte) (ytlogs.AviaWizardQueryRecord, error) {
	var record ytlogs.AviaWizardQueryRecord

	err := json.Unmarshal(message, &record)
	if err != nil {
		return record, xerrors.Errorf(
			"cannot unmarshal wizardSearches: %s: %w",
			string(message),
			err,
		)
	}

	return record, nil
}

func (c *AviaWizardQueryLogCollector) buildEventKey(event events.AviaUserSearch) string {
	return fmt.Sprintf("c%d_c%d", event.SettlementFromID, event.SettlementToID)
}
