package collectors

import (
	"context"
	"encoding/base64"
	"encoding/json"
	"fmt"
	"time"

	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/travel/avia/personalization/internal/auth"
	"a.yandex-team.ru/travel/avia/personalization/internal/consts"
	"a.yandex-team.ru/travel/avia/personalization/internal/events"
	"a.yandex-team.ru/travel/avia/personalization/internal/metrics"
	"a.yandex-team.ru/travel/avia/personalization/internal/tables"
	"a.yandex-team.ru/travel/library/go/strutil"
	"a.yandex-team.ru/travel/proto/cpa"
)

const (
	cpaOrdersLogMetricName        = "cpa-orders"
	cpaOrdersLogTimingsMetricName = "cpa-orders.timings"
)

type CPAOrdersEventBuilder interface {
	FromHotelsCPAOrder(record *cpa.TGenericOrderInfo, timestamp uint64) (events.HotelsCPAOrder, error)
	FromAviaCPAOrder(record *cpa.TGenericOrderInfo, timestamp uint64) (events.AviaCPAOrder, error)
}

type CPAOrdersCollector struct {
	logger       log.Logger
	eventsTable  *tables.UserEventsTable
	eventBuilder CPAOrdersEventBuilder
}

func NewCPAOrdersCollector(
	logger log.Logger,
	eventsTable *tables.UserEventsTable,
	eventBuilder CPAOrdersEventBuilder,
) *CPAOrdersCollector {
	return &CPAOrdersCollector{
		logger:       logger.WithName("CPAOrdersCollector"),
		eventsTable:  eventsTable,
		eventBuilder: eventBuilder,
	}
}

func (c *CPAOrdersCollector) OnMessage(message []byte) {
	startTime := time.Now()
	defer metrics.WriteTimings(cpaOrdersLogTimingsMetricName, startTime, map[string]string{"type": "batch"})

	parsedMessage, err := c.ParseMessage(message)

	if err != nil {
		c.logger.Error("Cannot read or handle cpa order message", log.Error(err), log.Any("message", message))
		return
	}
	c.handleOrderMessage(parsedMessage)
}

func (c *CPAOrdersCollector) OnError(persqueue.ReadMessageOrError) {
	metrics.IncCounterMetric(cpaOrdersLogMetricName, map[string]string{"type": "error"})
}

type CPAOrderMessage struct {
	Proto       string `json:"proto"`
	Timestamp   uint64 `json:"_timestamp"`
	MessageType string `json:"message_type"`
}

func (c *CPAOrdersCollector) ParseMessage(message []byte) (*CPAOrderMessage, error) {
	var parsedMessage CPAOrderMessage
	err := json.Unmarshal(message, &parsedMessage)
	if err != nil {
		c.logger.Error("cannot unmarshal travel cpa order message", log.Error(err), log.ByteString("rawMessage", message))
		return nil, fmt.Errorf("cannot unmarshal travel cpa order message: %s: %w", string(message), err)
	}
	decodedProto, err := base64.URLEncoding.DecodeString(parsedMessage.Proto)
	if err != nil {
		c.logger.Error("cannot decode base64")
		return nil, fmt.Errorf("cannot decode order proto: %w", err)
	}
	parsedMessage.Proto = string(decodedProto)
	return &parsedMessage, nil
}

func (c *CPAOrdersCollector) ParseOrderInfo(message []byte) (*cpa.TGenericOrderInfo, error) {
	orderInfo := cpa.TGenericOrderInfo{}
	err := proto.Unmarshal(message, &orderInfo)
	if err != nil {
		return nil, fmt.Errorf("cannot unmarshal travel cpa order proto: %w", err)
	}
	return &orderInfo, nil
}

func (c *CPAOrdersCollector) handleOrderMessage(parsedMessage *CPAOrderMessage) {
	startTime := time.Now()
	defer metrics.WriteTimings(cpaOrdersLogTimingsMetricName, startTime, map[string]string{"type": "all"})
	ctx := context.Background()
	orderInfo, err := c.ParseOrderInfo([]byte(parsedMessage.Proto))
	if err != nil {
		c.logger.Error("cannot read proto from cpa order info")
		return
	}
	c.logger.Debug(
		"Parsed Message",
		log.Any("message", parsedMessage),
	)
	if !c.validateOrderInfo(orderInfo) {
		return
	}
	event, err := c.buildEvent(orderInfo, parsedMessage.Timestamp)
	if err != nil {
		return
	}
	ctx = ctxlog.WithFields(
		ctx,
		log.UInt8("service", event.Service),
		log.UInt8("eventType", event.EventType),
		log.UInt32("createdAt", event.CreatedAt),
		log.String("authValue", event.AuthValue),
		log.UInt8("authType", event.AuthType),
	)
	defer metrics.WriteTimings(cpaOrdersLogTimingsMetricName, startTime, map[string]string{"type": "to_be_saved"})

	err = c.eventsTable.Upsert(ctx, event)
	if err != nil {
		ctxlog.Error(ctx, c.logger, "Cannot save message to YDB", log.Error(err))
		metrics.IncCounterMetric(cpaOrdersLogMetricName, map[string]string{"type": "error"})
		return
	}
	ctxlog.Info(ctx, c.logger, "CPA Order Event saved")
	metrics.IncCounterMetric(cpaOrdersLogMetricName, map[string]string{"type": "success"})
}

func (c *CPAOrdersCollector) extractYandexUID(orderInfo *cpa.TGenericOrderInfo) string {
	if orderInfo.GetLabel() == nil || orderInfo.GetLabel().GetLabelAvia() == nil && orderInfo.GetLabel().GetLabelHotels() == nil {
		return ""
	}
	labelAvia := orderInfo.GetLabel().GetLabelAvia()
	labelHotels := orderInfo.GetLabel().GetLabelHotels()
	return strutil.Coalesce(labelAvia.GetYandexUid(), labelHotels.GetYandexUid())
}

func (c *CPAOrdersCollector) extractPassportID(orderInfo *cpa.TGenericOrderInfo) string {
	if orderInfo.GetLabel() == nil || orderInfo.GetLabel().GetLabelAvia() == nil && orderInfo.GetLabel().GetLabelHotels() == nil {
		return ""
	}
	labelAvia := orderInfo.GetLabel().GetLabelAvia()
	labelHotels := orderInfo.GetLabel().GetLabelHotels()
	return strutil.Coalesce(labelAvia.GetPassportId(), labelHotels.GetPassportUid())
}

func (c *CPAOrdersCollector) validateOrderInfo(orderInfo *cpa.TGenericOrderInfo) bool {
	label := orderInfo.GetLabel()
	if label == nil || label.GetLabelHotels() == nil && label.GetLabelAvia() == nil {
		c.logger.Debug("Skip message without any label")
		return false
	}
	yandexUID := c.extractYandexUID(orderInfo)
	passportID := c.extractPassportID(orderInfo)
	if yandexUID == "" && passportID == "" {
		c.logger.Debug("Skip message without YandexUID and PassportID")
		return false
	}
	orderItems := orderInfo.GetOrderItems()
	if len(orderItems) == 0 {
		c.logger.Debug("Skip message without order items")
		return false
	}
	return c.validateAviaOrderInfo(orderItems[0].GetOrderInfoAvia()) || c.validateHotelOrderInfo(orderItems[0].GetOrderInfoHotels())
}

func (c *CPAOrdersCollector) validateAviaOrderInfo(orderItem *cpa.TOrderItemInfoAvia) bool {
	if orderItem == nil {
		return false
	}
	if orderItem.GetDateForward() == "" {
		c.logger.Debug("skip avia order without forward date")
		return false
	}
	if orderItem.GetOrigin() == "" || orderItem.GetDestination() == "" {
		c.logger.Debug("skip avia order without origin/destination")
	}
	return true
}

func (c *CPAOrdersCollector) validateHotelOrderInfo(orderItem *cpa.TOrderItemInfoHotels) bool {
	if orderItem == nil {
		return false
	}
	if orderItem.GetCheckIn() == "" || orderItem.GetCheckOut() == "" {
		c.logger.Debug("skip hotels order without check-in/check-out date")
		return false
	}
	return true
}

func (c *CPAOrdersCollector) buildEvent(orderInfo *cpa.TGenericOrderInfo, timestamp uint64) (tables.UserEventEntry, error) {
	if orderInfo.GetLabel().GetLabelHotels() != nil {
		return c.buildHotelOrderEvent(orderInfo, timestamp)
	} else if orderInfo.GetLabel().GetLabelAvia() != nil {
		return c.buildAviaOrderEvent(orderInfo, timestamp)
	}
	c.logger.Warn("got cpa order of unknown type", log.String("partner_order_id", orderInfo.PartnerOrderId), log.String("partner_name", orderInfo.PartnerName))
	return tables.UserEventEntry{}, fmt.Errorf("got cpa order of unknown type")
}

func (c *CPAOrdersCollector) buildHotelOrderEvent(orderInfo *cpa.TGenericOrderInfo, timestamp uint64) (tables.UserEventEntry, error) {
	event, err := c.eventBuilder.FromHotelsCPAOrder(orderInfo, timestamp)
	ctx := ctxlog.WithFields(
		context.Background(),
		log.String("partner_order_id", orderInfo.GetPartnerOrderId()),
		log.String("partner_name", orderInfo.GetPartnerName()),
	)
	if err != nil {
		ctxlog.Error(ctx, c.logger, "cannot prepare DB event", log.Error(err))
		return tables.UserEventEntry{}, fmt.Errorf("cannot prepare DB event: %w", err)
	}
	if event.YandexUID == "" && event.PassportID == "" {
		const noAuthData = "order without passport_id and yandex_uid"
		ctxlog.Error(ctx, c.logger, noAuthData)
		return tables.UserEventEntry{}, fmt.Errorf(noAuthData)
	}
	if event.SettlementToID == 0 {
		message := "cannot find hotel settlement"
		ctxlog.Warn(ctx, c.logger, message)
		return tables.UserEventEntry{}, fmt.Errorf(message)
	}
	eventData, err := json.Marshal(event)
	if err != nil {
		return tables.UserEventEntry{}, nil
	}
	authType := auth.TypeYandexUID
	authValue := event.YandexUID
	if event.PassportID != "" {
		authType = auth.TypePassportID
		authValue = event.PassportID
	}

	return tables.UserEventEntry{
		AuthType:  authType,
		AuthValue: authValue,
		Service:   consts.HotelsServiceID,
		EventType: consts.EventTypeOrder,
		EventKey:  fmt.Sprintf("hotel_at_c%d", event.SettlementToID),
		EventData: string(eventData),
		CreatedAt: event.Unixtime,
		ExpiresAt: event.Unixtime + uint32(consts.EventTTL.Milliseconds()),
	}, nil
}

func (c *CPAOrdersCollector) buildAviaOrderEvent(orderInfo *cpa.TGenericOrderInfo, timestamp uint64) (tables.UserEventEntry, error) {
	event, err := c.eventBuilder.FromAviaCPAOrder(orderInfo, timestamp)
	ctx := ctxlog.WithFields(
		context.Background(),
		log.String("partner_order_id", orderInfo.GetPartnerOrderId()),
		log.String("partner_name", orderInfo.GetPartnerName()),
	)
	if err != nil {
		ctxlog.Error(ctx, c.logger, "cannot prepare DB event", log.Error(err))
		return tables.UserEventEntry{}, fmt.Errorf("cannot prepare DB event: %w", err)
	}
	if event.DateForward == "" {
		message := "cannot find date forward"
		ctxlog.Warn(ctx, c.logger, message)
		return tables.UserEventEntry{}, fmt.Errorf(message)
	}
	if event.SettlementFromID == 0 {
		message := "cannot find origin settlement"
		ctxlog.Warn(ctx, c.logger, message)
		return tables.UserEventEntry{}, fmt.Errorf(message)
	}
	if event.SettlementToID == 0 {
		message := "cannot find destination settlement"
		ctxlog.Warn(ctx, c.logger, message)
		return tables.UserEventEntry{}, fmt.Errorf(message)
	}

	eventData, err := json.Marshal(event)
	if err != nil {
		c.logger.Error("failed to marshal avia cpa order event to json", log.Error(err))
		return tables.UserEventEntry{}, err
	}
	authType := auth.TypeYandexUID
	authValue := event.YandexUID
	if event.PassportID != "" {
		authType = auth.TypePassportID
		authValue = event.PassportID
	}
	return tables.UserEventEntry{
		AuthType:  authType,
		AuthValue: authValue,
		Service:   consts.AviaServiceID,
		EventType: consts.EventTypeOrder,
		EventKey:  fmt.Sprintf("c%d_c%d", event.SettlementFromID, event.SettlementToID),
		EventData: string(eventData),
		CreatedAt: event.Unixtime,
		ExpiresAt: event.Unixtime + uint32(consts.EventTTL.Milliseconds()),
	}, err
}
