package collectors

import (
	"context"
	"encoding/json"
	"fmt"
	"time"

	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/personalization/internal/auth"
	"a.yandex-team.ru/travel/avia/personalization/internal/consts"
	"a.yandex-team.ru/travel/avia/personalization/internal/events"
	"a.yandex-team.ru/travel/avia/personalization/internal/metrics"
	"a.yandex-team.ru/travel/avia/personalization/internal/tables"
	"a.yandex-team.ru/travel/avia/personalization/internal/ytlogs"
)

const (
	travelHotelsOfferCacheLogMetricName        = "travel-hotels-offercache-log"
	travelHotelsOfferCacheLogTimingsMetricName = "travel-hotels-offercache-log.timings"
)

type TravelHotelsOfferCacheLogCollectorConfig struct {
	OnlyClientIDs     []string `yaml:"only_client_ids" config:"HOTELS_OFFER_CACHE_LOG_COLLECTOR_ONLY_CLIENT_IDS"`
	ExcludedClientIDs []string `json:"excluded_client_ids" config:"HOTELS_OFFER_CACHE_LOG_COLLECTOR_EXCLUDED_CLIENT_IDS"`
}

type TravelHotelsOfferCacheLogEventBuilder interface {
	FromTravelOfferCache(record ytlogs.TravelHotelsOffercacheRecord) (events.HotelsUserSearch, error)
}

type TravelHotelsOfferCacheLogCollector struct {
	logger       log.Logger
	config       TravelHotelsOfferCacheLogCollectorConfig
	eventsTable  *tables.UserEventsTable
	eventBuilder TravelHotelsOfferCacheLogEventBuilder
}

func NewTravelHotelsOfferCacheLogCollector(
	logger log.Logger,
	config TravelHotelsOfferCacheLogCollectorConfig,
	eventsTable *tables.UserEventsTable,
	eventBuilder TravelHotelsOfferCacheLogEventBuilder,
) *TravelHotelsOfferCacheLogCollector {
	return &TravelHotelsOfferCacheLogCollector{
		logger:       logger.WithName("TravelHotelsOfferCacheLogCollector"),
		config:       config,
		eventsTable:  eventsTable,
		eventBuilder: eventBuilder,
	}
}

func (c *TravelHotelsOfferCacheLogCollector) OnMessage(message []byte) {
	startTime := time.Now()
	defer metrics.WriteTimings(travelHotelsOfferCacheLogTimingsMetricName, startTime, map[string]string{"type": "all"})

	parsedMessage, err := c.ParseMessage(message)
	if err != nil {
		c.logger.Error(
			"Cannot read or handle status message",
			log.Error(err),
			log.Any("message", message),
		)
		return
	}

	c.logger.Debug(
		"Parsed Message",
		log.Any("message", parsedMessage),
	)
	if parsedMessage.Req.YandexUID == "" && parsedMessage.Req.PassportUID == "" {
		c.logger.Info("Skip message without YandexUID and PassportID")
		return
	}
	ctx := ctxlog.WithFields(
		context.Background(),
		log.String("yandexuid", parsedMessage.Req.YandexUID),
		log.String("passportId", parsedMessage.Req.PassportUID),
	)
	if !parsedMessage.Req.UseSearcher {
		ctxlog.Info(ctx, c.logger, "Skip message without UseSearcher")
		return
	}
	if parsedMessage.Req.RequestID != nil && *parsedMessage.Req.RequestID != 0 {
		ctxlog.Info(ctx, c.logger, "Skip message with RequestId != 0")
		return
	}
	if parsedMessage.Info != nil {
		ctx = ctxlog.WithFields(ctx, log.String("clientID", parsedMessage.Info.OfferCacheClientID))
		if !c.checkOfferCacheClientID(ctx, parsedMessage.Info.OfferCacheClientID) {
			return
		}
	}
	if parsedMessage.Req.RequestID != nil {
		ctx = ctxlog.WithFields(ctx, log.Int("requestID", *parsedMessage.Req.RequestID))
	}
	if len(parsedMessage.Resp.Hotels) == 0 {
		ctxlog.Info(ctx, c.logger, "Skip message without hotels in response")
		return
	}
	event, err := c.eventBuilder.FromTravelOfferCache(parsedMessage)
	if err != nil {
		ctxlog.Error(ctx, c.logger, "Cannot prepare DB event", log.Error(err))
		return
	}

	if event.SettlementToID == 0 {
		ctxlog.Info(ctx, c.logger, "Cannot find hotel settlement")
		return
	}
	serializedEvent, err := json.Marshal(event)
	if err != nil {
		ctxlog.Error(ctx, c.logger, "Cannot marshal HotelsUserSearch message", log.Any("message", message))
		return
	}
	defer metrics.WriteTimings(travelHotelsOfferCacheLogTimingsMetricName, startTime, map[string]string{"type": "to_be_saved"})
	authType := auth.TypeYandexUID
	authValue := event.YandexUID
	if event.PassportID != "" {
		authType = auth.TypePassportID
		authValue = event.PassportID
	}
	eventsTableEntry := tables.UserEventEntry{
		AuthType:  authType,
		AuthValue: authValue,
		Service:   consts.HotelsServiceID,
		EventType: consts.EventTypeSearch,
		EventKey:  fmt.Sprintf("hotel_at_c%d", event.SettlementToID),
		EventData: string(serializedEvent),
		CreatedAt: event.Unixtime,
		ExpiresAt: event.Unixtime + uint32(consts.EventTTL.Milliseconds()),
	}
	err = c.eventsTable.Upsert(ctx, eventsTableEntry)
	if err != nil {
		ctxlog.Error(
			ctx,
			c.logger,
			"Cannot save message to YDB", log.Error(err),
			log.UInt8("service", consts.HotelsServiceID),
			log.UInt8("eventType", consts.EventTypeSearch),
		)
		metrics.IncCounterMetric(travelHotelsOfferCacheLogMetricName, map[string]string{"type": "error"})
		return
	}
	ctxlog.Info(
		ctx, c.logger,
		"HotelsSearch Event saved",
		log.UInt8("service", consts.HotelsServiceID),
		log.UInt8("eventType", consts.EventTypeSearch),
		log.UInt32("eventtime", event.Unixtime),
	)
	metrics.IncCounterMetric(travelHotelsOfferCacheLogMetricName, map[string]string{"type": "success"})
}

func (c *TravelHotelsOfferCacheLogCollector) OnError(decompressedMessage persqueue.ReadMessageOrError) {
	metrics.IncCounterMetric(travelHotelsOfferCacheLogMetricName, map[string]string{"type": "error"})
}

func (c *TravelHotelsOfferCacheLogCollector) ParseMessage(message []byte) (ytlogs.TravelHotelsOffercacheRecord, error) {
	var record ytlogs.TravelHotelsOffercacheRecord
	err := json.Unmarshal(message, &record)
	if err != nil {
		return record, xerrors.Errorf(
			"Cannot unmarshal travelOffers: %s: %w",
			string(message),
			err,
		)
	}

	return record, nil
}

func (c *TravelHotelsOfferCacheLogCollector) checkOfferCacheClientID(ctx context.Context, clientID string) bool {
	checkOnly := func() bool {
		if len(c.config.OnlyClientIDs) > 0 {
			for _, pattern := range c.config.OnlyClientIDs {
				if clientID == pattern {
					return true
				}
			}
			ctxlog.Info(
				ctx,
				c.logger,
				"Skip message with unexpected OfferCacheClientID",
				log.Strings("onlyClientIDs", c.config.OnlyClientIDs),
			)
			return false
		}
		return true
	}

	checkExcluded := func() bool {
		if len(c.config.ExcludedClientIDs) > 0 {
			for _, pattern := range c.config.ExcludedClientIDs {
				if clientID == pattern {
					ctxlog.Info(
						ctx,
						c.logger,
						"Skip message with unexpected OfferCacheClientID",
						log.Strings("excludedClientIDs", c.config.ExcludedClientIDs))
					return false
				}
			}
		}
		return true
	}
	return checkOnly() && checkExcluded()
}
