package collectors

import (
	"context"
	"encoding/json"
	"fmt"
	"time"

	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/xerrors"
	portalproto "a.yandex-team.ru/travel/api/proto/hotels_portal"
	"a.yandex-team.ru/travel/avia/personalization/internal/auth"
	"a.yandex-team.ru/travel/avia/personalization/internal/consts"
	"a.yandex-team.ru/travel/avia/personalization/internal/events"
	"a.yandex-team.ru/travel/avia/personalization/internal/metrics"
	"a.yandex-team.ru/travel/avia/personalization/internal/tables"
)

const (
	travelHotelsSearchLogMetricName        = "travel-hotels-search-log"
	travelHotelsSearchLogTimingsMetricName = "travel-hotels-search-log.timings"
)

type TravelHotelsSearchLogEventBuilder interface {
	FromTravelHotelsSearch(record *portalproto.THotelsSearchLogRecord) (events.HotelsUserSearch, error)
}

type TravelHotelsSearchLogCollector struct {
	logger       log.Logger
	eventsTable  *tables.UserEventsTable
	eventBuilder TravelHotelsSearchLogEventBuilder
}

func NewTravelHotelsSearchLogCollector(
	logger log.Logger,
	eventsTable *tables.UserEventsTable,
	eventBuilder TravelHotelsSearchLogEventBuilder,
) *TravelHotelsSearchLogCollector {
	return &TravelHotelsSearchLogCollector{
		logger:       logger.WithName("TravelHotelsSearchLogCollector"),
		eventsTable:  eventsTable,
		eventBuilder: eventBuilder,
	}
}

func (c *TravelHotelsSearchLogCollector) OnMessage(message []byte) {
	startTime := time.Now()
	defer metrics.WriteTimings(travelHotelsSearchLogTimingsMetricName, startTime, map[string]string{"type": "all"})

	parsedMessage, err := c.ParseMessage(message)
	if err != nil {
		c.logger.Error(
			"Cannot read or handle status message",
			log.Error(err),
			log.Any("message", message),
		)
		return
	}

	c.logger.Debug(
		"Parsed Message",
		log.Any("message", parsedMessage),
	)
	yandexUID := parsedMessage.RequestHeaders.GetYandexUid()
	if yandexUID == "" {
		c.logger.Info("Skip message without YandexUID")
		return
	}
	ctx := ctxlog.WithFields(context.Background(), log.String("yandexuid", yandexUID))
	if parsedMessage.GetResponse().GetFoundHotelCount() == 0 {
		ctxlog.Info(ctx, c.logger, "Skip message without hotels in response")
		return
	}
	event, err := c.eventBuilder.FromTravelHotelsSearch(parsedMessage)
	if err != nil {
		ctxlog.Error(ctx, c.logger, "Cannot prepare DB event", log.Error(err))
		return
	}

	if event.SettlementToID == 0 {
		ctxlog.Info(ctx, c.logger, "Cannot find hotel settlement")
		return
	}
	serializedEvent, err := json.Marshal(event)
	if err != nil {
		ctxlog.Error(ctx, c.logger, "Cannot marshal HotelsUserSearch message", log.Any("message", message))
		return
	}
	defer metrics.WriteTimings(travelHotelsSearchLogTimingsMetricName, startTime, map[string]string{"type": "to_be_saved"})
	authType := auth.TypeYandexUID
	authValue := event.YandexUID
	if event.PassportID != "" {
		authType = auth.TypePassportID
		authValue = event.PassportID
	}
	eventsTableEntry := tables.UserEventEntry{
		AuthType:  authType,
		AuthValue: authValue,
		Service:   consts.HotelsServiceID,
		EventType: consts.EventTypeSearch,
		EventKey:  fmt.Sprintf("hotel_at_c%d", event.SettlementToID),
		EventData: string(serializedEvent),
		CreatedAt: event.Unixtime,
		ExpiresAt: event.Unixtime + uint32(consts.EventTTL.Milliseconds()),
	}
	err = c.eventsTable.Upsert(ctx, eventsTableEntry)
	if err != nil {
		ctxlog.Error(
			ctx,
			c.logger,
			"Cannot save message to YDB", log.Error(err),
			log.UInt8("service", consts.HotelsServiceID),
			log.UInt8("eventType", consts.EventTypeSearch),
		)
		metrics.IncCounterMetric(travelHotelsSearchLogMetricName, map[string]string{"type": "error"})
		return
	}
	ctxlog.Info(
		ctx, c.logger,
		"HotelsSearch Event saved",
		log.UInt8("service", consts.HotelsServiceID),
		log.UInt8("eventType", consts.EventTypeSearch),
		log.UInt32("eventtime", event.Unixtime),
	)
	metrics.IncCounterMetric(travelHotelsSearchLogMetricName, map[string]string{"type": "success"})
}

func (c *TravelHotelsSearchLogCollector) OnError(decompressedMessage persqueue.ReadMessageOrError) {
	metrics.IncCounterMetric(travelHotelsSearchLogMetricName, map[string]string{"type": "error"})
}

func (c *TravelHotelsSearchLogCollector) ParseMessage(message []byte) (*portalproto.THotelsSearchLogRecord, error) {
	record := new(portalproto.THotelsSearchLogRecord)
	err := json.Unmarshal(message, record)
	if err != nil {
		return record, xerrors.Errorf(
			"Cannot unmarshal travelOffers: %s: %w",
			string(message),
			err,
		)
	}

	return record, nil
}
