package extractors

import (
	"fmt"
	"strconv"
	"strings"

	"a.yandex-team.ru/travel/avia/personalization/internal/caches/references"
)

type SettlementExtractor struct {
	settlementIDByStationID *StationIDToSettlementIDMapper
	references              *references.Registry
}

func NewSettlementExtractor(settlementIDByStationID *StationIDToSettlementIDMapper, references *references.Registry) *SettlementExtractor {
	return &SettlementExtractor{settlementIDByStationID: settlementIDByStationID, references: references}
}

func (s *SettlementExtractor) GetSettlementIDByCode(code string) (int32, error) {
	if settlementID, err := s.getBySettlementCode(code); err == nil {
		return settlementID, nil
	}
	return s.getByStationCode(code)
}

func (s *SettlementExtractor) GetSettlementIDByPointKey(pointKey string) (int32, error) {
	if pointKey == "" {
		return 0, fmt.Errorf("empty point key")
	}
	if strings.HasPrefix(pointKey, "c") {
		settlementID, err := strconv.Atoi(pointKey[1:])
		return int32(settlementID), err
	}
	stationID, err := strconv.Atoi(pointKey[1:])
	if err != nil {
		return 0, err
	}
	settlementID, found := s.settlementIDByStationID.Map(int32(stationID))
	if !found {
		return 0, NewErrStationWithoutSettlement(stationID)
	}
	return settlementID, nil
}

func (s *SettlementExtractor) getBySettlementCode(settlementCode string) (int32, error) {
	settlement, found := s.references.Settlements.GetByCode(settlementCode)
	if !found {
		return 0, fmt.Errorf("unknown settlement code")
	}
	return settlement.Id, nil
}

func (s *SettlementExtractor) getByStationCode(stationCode string) (int32, error) {
	stationID, found := s.references.StationCodes.GetStationIDByCode(stationCode)
	if !found {
		return 0, fmt.Errorf("unknown station code")
	}
	settlementID, found := s.settlementIDByStationID.Map(stationID)
	if !found {
		return 0, fmt.Errorf("station has no settlement")
	}
	return settlementID, nil
}
