package repositories

import (
	"compress/zlib"
	"fmt"
	"io/ioutil"
	"os"
	"path/filepath"
	"runtime"
	"sync"
	"sync/atomic"

	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/library/go/core/log"
	hotelsreference "a.yandex-team.ru/travel/avia/personalization/hotels_reference/proto"
	"a.yandex-team.ru/travel/avia/personalization/internal/caches/status"
)

type HotelsReferenceConfig struct {
	ResourcePath        string `config:"HOTELS_REFERENCE_RESOURCE_PATH" yaml:"resource_path"`
	UseDynamicResources bool   `yaml:"use_dynamic_resources"`
}

var DefaultHotelsReferenceConfig = HotelsReferenceConfig{
	ResourcePath:        "",
	UseDynamicResources: false,
}

type Hotel struct {
	Permalink int     `yson:"permalink"`
	GeoID     int     `yson:"geoid"`
	Latitude  float32 `yson:"lat"`
	Longitude float32 `yson:"lon"`
}

type Hotels []Hotel
type hotelByPermalinkType map[int]Hotel

type HotelsProvider struct {
	logger     log.Logger
	updateLock *sync.Mutex

	config *HotelsReferenceConfig

	status           status.CacheStatus
	hotels           Hotels
	hotelByPermalink atomic.Value
}

func NewHotelsProvider(logger log.Logger, config *HotelsReferenceConfig) *HotelsProvider {
	provider := &HotelsProvider{
		logger:           logger,
		config:           config,
		status:           status.NotStarted,
		hotels:           make(Hotels, 0),
		hotelByPermalink: atomic.Value{},
		updateLock:       &sync.Mutex{},
	}
	provider.hotelByPermalink.Store(make(hotelByPermalinkType))
	return provider
}

func (provider *HotelsProvider) Ready() bool {
	return provider.status == status.Finished
}

func (provider *HotelsProvider) GetHotelByPermalink(permalink int) *Hotel {
	if !provider.Ready() {
		return nil
	}
	h, ok := provider.hotelByPermalink.Load().(hotelByPermalinkType)[permalink]
	if ok {
		return &h
	} else {
		return nil
	}
}

func (provider *HotelsProvider) Precache() error {
	provider.updateLock.Lock()
	defer provider.updateLock.Unlock()
	provider.logger.Info("start precaching hotels")
	defer provider.logger.Info("finish precaching hotels")
	provider.status = status.InProgress
	err := provider.precache()
	if err != nil {
		provider.logger.Error(fmt.Sprintf("couldn't precache hotels: %+v", err))
		provider.status = status.NotStarted
		return err
	}
	provider.status = status.Finished

	return nil
}

func (provider *HotelsProvider) precache() error {
	hotels, err := provider.readFromFile()
	if err != nil {
		return err
	}

	hotelByPermalink := make(hotelByPermalinkType, len(hotels))
	for _, hotel := range hotels {
		hotelByPermalink[hotel.Permalink] = hotel
	}
	provider.hotelByPermalink.Store(hotelByPermalink)
	// caching allocates a lot of memory
	// so it's necessary to force garbage collecting
	runtime.GC()
	return nil
}

func (provider *HotelsProvider) readFromFile() (Hotels, error) {
	resourcePath, err := getResourcePath(provider.config)
	if err != nil {
		return nil, err
	}
	file, err := os.Open(resourcePath)
	if err != nil {
		return nil, err
	}
	defer file.Close()
	zlibReader, err := zlib.NewReader(file)
	if err != nil {
		return nil, err
	}
	defer zlibReader.Close()
	protoBytes, err := ioutil.ReadAll(zlibReader)
	if err != nil {
		return nil, err
	}
	const approximateHotelsCount = 9 * 1000 * 1000
	hotels := make(Hotels, 0, approximateHotelsCount)
	hotelsProto := hotelsreference.Hotels{}
	if err := proto.Unmarshal(protoBytes, &hotelsProto); err != nil {
		return nil, err
	}
	for _, hotelProto := range hotelsProto.Hotels {
		hotels = append(
			hotels, Hotel{
				Permalink: int(hotelProto.Permalink),
				GeoID:     int(hotelProto.GeoId),
				Latitude:  hotelProto.Latitude,
				Longitude: hotelProto.Longitude,
			},
		)
	}
	return hotels, nil
}

func getResourcePath(config *HotelsReferenceConfig) (string, error) {
	resourcePath := config.ResourcePath
	if config.UseDynamicResources {
		return filepath.EvalSymlinks(resourcePath)
	}
	return resourcePath, nil
}
