package dynamicresources

import (
	"encoding/json"
	"net/http"

	"github.com/go-chi/chi/v5"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/httputil/headers"
)

type Config struct {
	Addr              string
	UpdateDictsRoute  string
	UpdateHotelsRoute string
}

var DefaultConfig = Config{
	Addr:              "[::]:9006",
	UpdateDictsRoute:  "/update_dicts",
	UpdateHotelsRoute: "/update_hotels",
}

type Service struct {
	logger         log.Logger
	onUpdateDicts  func() error
	onUpdateHotels func() error
	config         Config
}

func NewService(logger log.Logger, config Config, onUpdateDicts, onUpdateHotels func() error) *Service {
	return &Service{
		logger:         logger.WithName("DynamicResourcesService"),
		config:         config,
		onUpdateDicts:  onUpdateDicts,
		onUpdateHotels: onUpdateHotels,
	}
}

func (s *Service) BackroundRun() {
	router := chi.NewRouter()
	router.Get(s.config.UpdateDictsRoute, s.updateDictsHandler)
	router.Get(s.config.UpdateHotelsRoute, s.updateHotelsHandler)
	go func() {
		_ = http.ListenAndServe(s.config.Addr, router)
	}()
}

type updateResourcesResponse struct {
	Status string `json:"status"`
	Error  string `json:"error,omitempty"`
}

var successResponse, _ = json.Marshal(updateResourcesResponse{Status: "success"})

func (s *Service) updateDictsHandler(w http.ResponseWriter, r *http.Request) {
	s.logger.Info("update shared dicts dynamic resource request")
	w.WriteHeader(http.StatusOK)
	w.Header().Set(headers.ContentTypeKey, headers.TypeApplicationJSON.String())
	if err := s.onUpdateDicts(); err == nil {
		_, _ = w.Write(successResponse)
		s.logger.Info("shared dicts dynamic resource updated successfully")
	} else {
		s.logger.Error("failed to update shared dicts", log.Error(err))
		response, _ := json.Marshal(updateResourcesResponse{Status: "failed", Error: err.Error()})
		_, _ = w.Write(response)
	}
}

func (s *Service) updateHotelsHandler(w http.ResponseWriter, r *http.Request) {
	s.logger.Info("update hotels dynamic resource request")
	w.WriteHeader(http.StatusOK)
	w.Header().Set(headers.ContentTypeKey, headers.TypeApplicationJSON.String())
	if err := s.onUpdateHotels(); err == nil {
		_, _ = w.Write(successResponse)
		s.logger.Info("hotels dynamic resource updated successfully")
	} else {
		s.logger.Error("failed to update hotels reference", log.Error(err))
		response, _ := json.Marshal(updateResourcesResponse{Status: "failed", Error: err.Error()})
		_, _ = w.Write(response)
	}
}
