package eventscollector

import (
	"context"
	"sync"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/personalization/internal/builders"
	"a.yandex-team.ru/travel/avia/personalization/internal/collectors"
	"a.yandex-team.ru/travel/avia/personalization/internal/logbroker"
	"a.yandex-team.ru/travel/avia/personalization/internal/tables"
)

type Service struct {
	logger log.Logger
	cfg    Config

	aviaUserSearchLogCollector   *collectors.AviaUserSearchLogCollector
	cpaOrdersCollector           *collectors.CPAOrdersCollector
	aviaWizardQueryLogCollector  *collectors.AviaWizardQueryLogCollector
	hotelsSearchLogCollector     *collectors.TravelHotelsSearchLogCollector
	hotelsOfferCacheLogCollector *collectors.TravelHotelsOfferCacheLogCollector

	aviaUserSearchLogConsumer   *logbroker.MultiEndpointConsumer
	cpaOrdersConsumer           *logbroker.MultiEndpointConsumer
	aviaWizardQueryLogConsumer  *logbroker.MultiEndpointConsumer
	hotelsSearchLogConsumer     *logbroker.MultiEndpointConsumer
	hotelsOfferCacheLogConsumer *logbroker.MultiEndpointConsumer
}

func NewService(
	logbrokerConfig logbroker.LogbrokerConfig,
	eventsCollectorConfig Config,
	logger log.Logger,
	eventsTable *tables.UserEventsTable,
	eventBuilder *builders.EventBuilder,
) *Service {
	return &Service{
		logger: logger.WithName("EventsCollector"),
		cfg:    eventsCollectorConfig,

		aviaUserSearchLogCollector:  collectors.NewAviaUserSearchLogCollector(logger, eventsTable, eventBuilder),
		cpaOrdersCollector:          collectors.NewCPAOrdersCollector(logger, eventsTable, eventBuilder),
		aviaWizardQueryLogCollector: collectors.NewAviaWizardQueryLogCollector(logger, eventsTable, eventBuilder),
		hotelsSearchLogCollector:    collectors.NewTravelHotelsSearchLogCollector(logger, eventsTable, eventBuilder),
		hotelsOfferCacheLogCollector: collectors.NewTravelHotelsOfferCacheLogCollector(
			logger,
			eventsCollectorConfig.TravelHotelsOfferCacheLogCollector,
			eventsTable,
			eventBuilder,
		),

		aviaUserSearchLogConsumer: logbroker.NewMultiEndpointConsumer(
			logger,
			eventsCollectorConfig.AviaUserSearchLog,
			logbrokerConfig.Token,
		),
		cpaOrdersConsumer: logbroker.NewMultiEndpointConsumer(
			logger,
			eventsCollectorConfig.CPAOrders,
			logbrokerConfig.Token,
		),
		aviaWizardQueryLogConsumer: logbroker.NewMultiEndpointConsumer(
			logger,
			eventsCollectorConfig.AviaWizardQueryLog,
			logbrokerConfig.Token,
		),
		hotelsSearchLogConsumer: logbroker.NewMultiEndpointConsumer(
			logger,
			eventsCollectorConfig.TravelHotelsSearchLog,
			logbrokerConfig.Token,
		),
		hotelsOfferCacheLogConsumer: logbroker.NewMultiEndpointConsumer(
			logger,
			eventsCollectorConfig.TravelHotelsOfferCacheLog,
			logbrokerConfig.Token,
		),
	}
}

func (s *Service) Run(ctx context.Context) {
	wg := sync.WaitGroup{}
	wg.Add(5)
	go func() {
		defer wg.Done()
		if s.cfg.CollectAviaUserSearch {
			s.aviaUserSearchLogConsumer.Run(ctx, s.aviaUserSearchLogCollector)
		} else {
			s.logger.Info("Collecting avia user search is disabled")
		}
	}()

	go func() {
		defer wg.Done()
		if s.cfg.CollectCPAOrders {
			s.cpaOrdersConsumer.Run(ctx, s.cpaOrdersCollector)
		} else {
			s.logger.Info("Collecting CPA orders is disabled")
		}
	}()

	go func() {
		defer wg.Done()
		if s.cfg.CollectAviaWizardQueryLog {
			s.aviaWizardQueryLogConsumer.Run(ctx, s.aviaWizardQueryLogCollector)
		} else {
			s.logger.Info("Collecting avia wizard query log is disabled")
		}
	}()

	go func() {
		defer wg.Done()
		if s.cfg.CollectHotelsOfferCacheLog {
			s.hotelsOfferCacheLogConsumer.Run(ctx, s.hotelsOfferCacheLogCollector)
		} else {
			s.logger.Info("Collecting hotels offer cache log is disabled")
		}
	}()

	go func() {
		defer wg.Done()
		if s.cfg.CollectHotelsSearchLog {
			s.hotelsSearchLogConsumer.Run(ctx, s.hotelsSearchLogCollector)
		} else {
			s.logger.Info("Collecting hotels portal log is disabled")
		}
	}()

	wg.Wait()
}

func (s *Service) HasEnabledSources() bool {
	return s.cfg.CollectAviaUserSearch ||
		s.cfg.CollectAviaWizardQueryLog ||
		s.cfg.CollectHotelsOfferCacheLog ||
		s.cfg.CollectCPAOrders ||
		s.cfg.CollectHotelsSearchLog
}

func (s *Service) IsReady() bool {
	return (!s.cpaOrdersConsumer.Running() || s.cpaOrdersConsumer.Ready()) &&
		(!s.aviaUserSearchLogConsumer.Running() || s.aviaUserSearchLogConsumer.Ready()) &&
		(!s.aviaWizardQueryLogConsumer.Running() || s.aviaWizardQueryLogConsumer.Ready()) &&
		(!s.hotelsOfferCacheLogConsumer.Running() || s.hotelsOfferCacheLogConsumer.Ready()) &&
		(!s.hotelsSearchLogConsumer.Running() || s.hotelsSearchLogConsumer.Ready())
}
