package personalsearch

import (
	"context"
	"encoding/json"
	"sort"
	"time"

	"github.com/jonboulle/clockwork"
	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/travel/avia/personalization/internal/caches/references"
	"a.yandex-team.ru/travel/avia/personalization/internal/consts"
	eventmodels "a.yandex-team.ru/travel/avia/personalization/internal/events"
	"a.yandex-team.ru/travel/avia/personalization/internal/lib"
	"a.yandex-team.ru/travel/avia/personalization/internal/services/personalsearch/models"
	"a.yandex-team.ru/travel/avia/personalization/internal/tables"
	"a.yandex-team.ru/travel/library/go/containers"
	"a.yandex-team.ru/travel/proto/cpa"
)

type EventsProvider struct {
	logger      log.Logger
	eventsTable *tables.UserEventsTable
	references  *references.Registry
	clock       clockwork.Clock
}

func NewEventsProvider(
	l log.Logger,
	eventsTable *tables.UserEventsTable,
	references *references.Registry,
	clock clockwork.Clock,
) *EventsProvider {
	service := &EventsProvider{
		logger:      l,
		eventsTable: eventsTable,
		references:  references,
		clock:       clock,
	}
	return service
}

func (s *EventsProvider) GetEvents(ctx context.Context, query *Query, geoID uint32) ([]models.Event, error) {
	span, _ := opentracing.StartSpanFromContext(ctx, "internal.services.personalsearch.EventsProvider:GetEvents")
	defer span.Finish()

	rawEventsByServiceType, err := s.getRawEvents(ctx, query)
	rawEvents := flattenEvents(rawEventsByServiceType)
	if err != nil {
		return nil, err
	}
	events := s.mapEvents(rawEvents, geoID)
	sort.SliceStable(
		events, func(i, j int) bool {
			return events[i].Timestamp > events[j].Timestamp
		},
	)
	events = removeEventsWithInvalidPoints(events)
	if query.RemoveUnconfirmedOrders {
		events = removeUnconfirmedOrders(events)
	}
	if query.DeduplicateByDirection {
		events = deduplicateByDirection(events, query)
	}
	if query.FillAviaReturnDate {
		events = fillAviaReturnDate(events)
	}
	if query.RemoveLongTermAviaOrders {
		events = removeLongTermAviaOrders(events)
	}
	if query.RemoveOneDayRoundTrip {
		events = removeOneDayRoundTrip(events)
	}
	events = filterOutdated(events, s.clock.Now())
	return getSortedEvents(events, query.TravelService), nil
}

func flattenEvents(entriesByServiceType map[tables.EventServiceTypeKey]tables.UserEventEntries) (result []tables.UserEventEntry) {
	for _, entries := range entriesByServiceType {
		for _, entry := range entries {
			result = append(result, entry)
		}
	}
	return result
}

func (s *EventsProvider) mapEvents(entries []tables.UserEventEntry, geoID uint32) []models.Event {
	events := make([]models.Event, 0, len(entries))
	for _, entry := range entries {
		events = append(events, s.entryToEvent(entry, geoID))
	}
	return events
}

func (s *EventsProvider) updateConfirmedOrders(confirmedOrderIDs containers.Set[string], order models.OrderInfo) {
	if order.ID == "" {
		return
	}
	if order.Status == cpa.EOrderStatus_OS_CONFIRMED.String() {
		confirmedOrderIDs.Add(order.ID)
	} else {
		confirmedOrderIDs.Remove(order.ID)
	}
}

func (s *EventsProvider) entryToEvent(element tables.UserEventEntry, geoID uint32) models.Event {
	logCtx := ctxlog.WithFields(
		context.Background(),
		log.UInt8("authType", element.AuthType),
		log.String("authValue", element.AuthValue),
		log.String("eventData", element.EventData),
		log.UInt8("eventService", element.Service),
		log.String("eventKey", element.EventKey),
		log.UInt8("eventType", element.EventType),
		log.Time("createdAt", time.Unix(int64(element.CreatedAt), 0)),
		log.Time("expiresAt", time.Unix(int64(element.ExpiresAt), 0)),
	)
	if element.Service == consts.AviaServiceID {
		entry, err := s.processAviaEvent(element)
		if err != nil {
			ctxlog.Error(logCtx, s.logger, "Unmarshal event error", log.Error(err))
		}
		return entry
	} else if element.Service == consts.HotelsServiceID {
		entry, err := s.processHotelEvent(element, geoID)
		if err != nil {
			ctxlog.Error(logCtx, s.logger, "Unmarshal event error", log.Error(err))
		}
		return entry
	} else {
		ctxlog.Error(logCtx, s.logger, "Unknown service ID")
		return models.Event{}
	}
}

func (s *EventsProvider) processAviaEvent(element tables.UserEventEntry) (result models.Event, err error) {
	result.Service = consts.AviaServiceName
	result.Timestamp = element.CreatedAt
	if element.EventType == consts.EventTypeSearch {
		var event eventmodels.AviaUserSearch
		err_ := json.Unmarshal([]byte(element.EventData), &event)
		if err_ != nil {
			return result, err_
		}
		result.PointFrom = s.mapPoint(event.SettlementFromID)
		result.PointTo = s.mapPoint(event.SettlementToID)
		result.When = event.When
		result.ReturnDate = event.ReturnDate
		result.Travelers = models.Travelers{
			Adults:       event.AdultSeats,
			Children:     event.ChildrenSeats,
			Infants:      event.InfantSeats,
			ChildrenAges: []int{},
		}
		result.AviaClass = event.Klass
	} else if element.EventType == consts.EventTypeOrder {
		var event eventmodels.AviaCPAOrder
		err_ := json.Unmarshal([]byte(element.EventData), &event)
		if err_ != nil {
			return result, err_
		}
		result.PointFrom = s.mapPoint(event.SettlementFromID)
		result.PointTo = s.mapPoint(event.SettlementToID)
		result.When = event.DateForward
		result.ReturnDate = event.DateBackward
		result.Travelers = models.Travelers{
			Adults:       event.AdultSeats,
			Children:     event.ChildrenSeats,
			Infants:      event.InfantSeats,
			ChildrenAges: []int{},
		}
		result.AviaClass = event.ServiceClass
		result.Order = models.OrderInfo{ID: event.OrderID, Status: event.Status}
	} else {
		s.logger.Warn("unknown event type", log.UInt8("eventType", element.EventType))
	}
	return result, err
}

func (s *EventsProvider) processHotelEvent(element tables.UserEventEntry, geoID uint32) (result models.Event, err error) {
	result.Service = consts.HotelsServiceName
	result.Timestamp = element.CreatedAt
	if element.EventType == consts.EventTypeSearch {
		var event eventmodels.HotelsUserSearch
		err_ := json.Unmarshal([]byte(element.EventData), &event)
		if err_ != nil {
			return result, err_
		}

		result.PointTo = s.mapPoint(event.SettlementToID)
		result.When = event.CheckInDate
		result.ReturnDate = event.CheckOutDate
		result.Travelers = models.Travelers{
			Adults:       event.AdultSeats,
			Children:     event.ChildrenSeats,
			Infants:      event.InfantSeats,
			ChildrenAges: getChildrenAges(event.Ages),
		}
	} else {
		var event eventmodels.HotelsCPAOrder
		err_ := json.Unmarshal([]byte(element.EventData), &event)
		if err_ != nil {
			return result, err_
		}

		result.PointTo = s.mapPoint(event.SettlementToID)
		result.When = event.CheckInDate
		result.ReturnDate = event.CheckOutDate
		result.Order = models.OrderInfo{ID: event.OrderID, Status: event.Status}
	}
	settlement, found := s.references.Settlements.GetByGeoID(int32(geoID))
	if !found {
		return
	}

	result.PointFrom = s.mapPoint(settlement.GetId())
	result.AviaClass = "economy"
	return result, err
}

func (s *EventsProvider) mapPoint(settlementID int32) models.GeoPoint {
	pointKey := lib.SettlementIDToPointKey(uint64(settlementID))
	return models.GeoPoint{PointKey: pointKey, GeoID: s.pointKeyToGeoID(pointKey)}
}

func (s *EventsProvider) pointKeyToGeoID(pointKey string) uint64 {
	settlementID := lib.PointKeyToID(pointKey)
	if settlement, found := s.references.Settlements.Get(int32(settlementID)); found {
		return uint64(settlement.GeoId)
	}
	return 0
}

func (s *EventsProvider) getRawEvents(ctx context.Context, query *Query) (map[tables.EventServiceTypeKey]tables.UserEventEntries, error) {
	ydbEntries, err := s.eventsTable.SelectByServiceTypePairs(ctx, query.AuthType, query.AuthValue, query.ServiceEventTypePairs)
	if err != nil {
		s.logger.Error("failed to get raw events", log.Error(err))
		return nil, err
	}
	return ydbEntries, nil
}
