package grpc

import (
	"context"
	"time"

	"github.com/opentracing/opentracing-go"
	"google.golang.org/grpc"

	"a.yandex-team.ru/library/go/core/log"
	pb "a.yandex-team.ru/travel/avia/personalization/api"
	personalsearchproto "a.yandex-team.ru/travel/avia/personalization/api/personal_search/v2"
	"a.yandex-team.ru/travel/avia/personalization/internal/auth"
	"a.yandex-team.ru/travel/avia/personalization/internal/consts"
	"a.yandex-team.ru/travel/avia/personalization/internal/metrics"
	"a.yandex-team.ru/travel/avia/personalization/internal/services/personalsearch"
	"a.yandex-team.ru/travel/avia/personalization/internal/tables"
)

const defaultAviaHistoryResultsLimit uint8 = 3

func (h *PersonalizationService) GetAviaHistory(ctx context.Context, request *personalsearchproto.TGetAviaHistoryRequestV2) (
	response *personalsearchproto.TGetPersonalSearchResponseV2,
	err error,
) {
	start := time.Now()
	defer metrics.WriteTimings("PersonalSearch", start, map[string]string{"request_type": "AviaHistory"})
	span, _ := opentracing.StartSpanFromContext(ctx, "internal.services.personalsearch.handler.GRPCPersonalSearchHandler:GetAviaHistory")
	defer span.Finish()

	var limit = defaultAviaHistoryResultsLimit
	if request.Limit != 0 {
		limit = uint8(request.Limit)
	}
	eventServiceTypePairs := tables.EventServiceTypeToLimit{
		tables.EventServiceTypeKey{Service: consts.AviaServiceID, Type: consts.EventTypeSearch}: limit,
	}
	var query *personalsearch.Query
	if request.GetPassportId() != "" {
		query = personalsearch.NewQuery(auth.TypePassportID, request.PassportId, eventServiceTypePairs, consts.AviaServiceName)
	} else {
		query = personalsearch.NewQuery(auth.TypeYandexUID, request.YandexUid, eventServiceTypePairs, consts.AviaServiceName)
	}

	events, err := h.eventsProvider.GetEvents(ctx, query, 0)

	if err != nil {
		h.logger.Error("Internal error", log.Error(err))
		return
	}

	response = &personalsearchproto.TGetPersonalSearchResponseV2{Entries: make([]*personalsearchproto.TPersonalSearchEntryV2, 0)}
	for _, v := range events {
		pbEntry := &personalsearchproto.TAviaEntry{
			PointFrom: &personalsearchproto.TGeoPoint{
				PointCode: v.PointFrom.PointKey,
			},
			PointTo: &personalsearchproto.TGeoPoint{
				PointCode: v.PointTo.PointKey,
			},
			When:       v.When,
			ReturnDate: v.ReturnDate,
			Travelers: &personalsearchproto.TTravelers{
				Adults:   int64(v.Travelers.Adults),
				Children: int64(v.Travelers.Children),
				Infants:  int64(v.Travelers.Infants),
			},
			AviaClass: v.AviaClass,
		}
		response.Entries = append(
			response.Entries,
			&personalsearchproto.TPersonalSearchEntryV2{Entry: &personalsearchproto.TPersonalSearchEntryV2_Avia{Avia: pbEntry}},
		)
	}
	return response, nil
}

func (h *PersonalizationService) GetPersonalSearch(ctx context.Context, request *pb.TGetPersonalSearchRequest) (
	response *pb.TGetPersonalSearchResponse,
	err error,
) {
	requestV2 := &personalsearchproto.TGetAviaHistoryRequestV2{YandexUid: request.YandexUid, Limit: 10}
	responseV2, err := h.GetAviaHistory(ctx, requestV2)
	if err != nil {
		return nil, err
	}
	response = &pb.TGetPersonalSearchResponse{Entries: make([]*pb.TPersonalSearchEntry, 0)}
	for _, v := range responseV2.Entries {
		pbEntry := &pb.TPersonalSearchEntry{
			PointFrom:  &pb.TGeoPoint{PointCode: v.GetAvia().PointFrom.PointCode},
			PointTo:    &pb.TGeoPoint{PointCode: v.GetAvia().PointTo.PointCode},
			When:       v.GetAvia().When,
			ReturnDate: v.GetAvia().ReturnDate,
			Travellers: &pb.TTravelers{
				Adults:   v.GetAvia().Travelers.Adults,
				Children: v.GetAvia().Travelers.Children,
				Infants:  v.GetAvia().Travelers.Infants,
			},
			AviaClass: v.GetAvia().AviaClass,
		}
		response.Entries = append(response.Entries, pbEntry)
	}
	return response, nil
}

func (h *PersonalizationService) GetOldServiceRegisterer() func(*grpc.Server) {
	return func(server *grpc.Server) {
		pb.RegisterPersonalSearchServiceV1Server(server, h)
	}
}
