package grpc

import (
	"context"
	"time"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	personalsearchproto "a.yandex-team.ru/travel/avia/personalization/api/personal_search/v2"
	"a.yandex-team.ru/travel/avia/personalization/internal/auth"
	"a.yandex-team.ru/travel/avia/personalization/internal/consts"
	"a.yandex-team.ru/travel/avia/personalization/internal/metrics"
	"a.yandex-team.ru/travel/avia/personalization/internal/services/personalsearch"
	"a.yandex-team.ru/travel/avia/personalization/internal/tables"
)

func (h *PersonalizationService) GetAviaSuggest(ctx context.Context, request *personalsearchproto.TGetAviaSuggestRequestV2) (
	response *personalsearchproto.TGetPersonalSearchResponseV2,
	err error,
) {
	start := time.Now()
	defer metrics.WriteTimings("PersonalSearch", start, map[string]string{"request_type": "AviaSuggest"})
	span, _ := opentracing.StartSpanFromContext(ctx, "internal.services.personalsearch.handler.PersonalizationService:GetAviaSuggest")
	defer span.Finish()

	eventTypeByService := tables.EventServiceTypeToLimit{
		tables.EventServiceTypeKey{Service: consts.AviaServiceID, Type: consts.EventTypeSearch}:  uint8(request.SearchesLimit),
		tables.EventServiceTypeKey{Service: consts.HotelsServiceID, Type: consts.EventTypeOrder}: uint8(request.OrdersLimit),
	}
	var query *personalsearch.Query
	if request.GetPassportId() != "" {
		query = personalsearch.NewQuery(auth.TypePassportID, request.PassportId, eventTypeByService, consts.AviaServiceName)
	} else {
		query = personalsearch.NewQuery(auth.TypeYandexUID, request.YandexUid, eventTypeByService, consts.AviaServiceName)
	}

	entries, err := h.eventsProvider.GetEvents(ctx, query, request.GeoId)

	if err != nil {
		h.logger.Error("Internal error", log.Error(err))
		return
	}

	response = &personalsearchproto.TGetPersonalSearchResponseV2{}
	for _, entry := range entries {
		pbEntry := &personalsearchproto.TAviaEntry{
			PointFrom: &personalsearchproto.TGeoPoint{
				PointCode: entry.PointFrom.PointKey,
				GeoId:     entry.PointFrom.GeoID,
			},
			PointTo: &personalsearchproto.TGeoPoint{
				PointCode: entry.PointTo.PointKey,
				GeoId:     entry.PointTo.GeoID,
			},
			When:       entry.When,
			ReturnDate: entry.ReturnDate,
			Travelers: &personalsearchproto.TTravelers{
				Adults:   int64(entry.Travelers.Adults),
				Children: int64(entry.Travelers.Children),
				Infants:  int64(entry.Travelers.Infants),
			},
			AviaClass: entry.AviaClass,
			Type:      getEventTypeProto(entry),
		}
		response.Entries = append(
			response.Entries,
			&personalsearchproto.TPersonalSearchEntryV2{Entry: &personalsearchproto.TPersonalSearchEntryV2_Avia{Avia: pbEntry}},
		)
	}
	return response, nil
}
