package grpc

import (
	"context"
	"time"

	"github.com/opentracing/opentracing-go"
	"google.golang.org/grpc"

	"a.yandex-team.ru/library/go/core/log"
	personalsearchproto "a.yandex-team.ru/travel/avia/personalization/api/personal_search/v2"
	"a.yandex-team.ru/travel/avia/personalization/internal/auth"
	"a.yandex-team.ru/travel/avia/personalization/internal/consts"
	"a.yandex-team.ru/travel/avia/personalization/internal/metrics"
	"a.yandex-team.ru/travel/avia/personalization/internal/services/personalsearch"
	"a.yandex-team.ru/travel/avia/personalization/internal/services/personalsearch/models"
	"a.yandex-team.ru/travel/avia/personalization/internal/tables"
)

type PersonalizationService struct {
	eventsProvider EventsBuilder
	logger         log.Logger
}

func NewPersonalizationService(eventsBuilder EventsBuilder, logger log.Logger) *PersonalizationService {
	return &PersonalizationService{eventsProvider: eventsBuilder, logger: logger}
}

func (h *PersonalizationService) GetHotelsSuggest(ctx context.Context, request *personalsearchproto.TGetHotelsSuggestRequestV2) (
	*personalsearchproto.TGetPersonalSearchResponseV2,
	error,
) {
	start := time.Now()
	defer metrics.WriteTimings("PersonalSearch", start, map[string]string{"request_type": "HotelsSuggest"})
	span, _ := opentracing.StartSpanFromContext(ctx, "internal.services.personalsearch.handler.PersonalizationService:GetHotelsSuggest")
	defer span.Finish()

	eventTypeByService := tables.EventServiceTypeToLimit{
		tables.EventServiceTypeKey{Service: consts.AviaServiceID, Type: consts.EventTypeOrder}:    uint8(request.OrdersLimit),
		tables.EventServiceTypeKey{Service: consts.HotelsServiceID, Type: consts.EventTypeSearch}: uint8(request.SearchesLimit),
	}
	var query *personalsearch.Query
	if request.GetPassportId() != "" {
		query = personalsearch.NewQuery(auth.TypePassportID, request.PassportId, eventTypeByService, consts.HotelsServiceName)
	} else {
		query = personalsearch.NewQuery(auth.TypeYandexUID, request.YandexUid, eventTypeByService, consts.HotelsServiceName)
	}
	query.FillAviaReturnDate = true
	query.RemoveLongTermAviaOrders = true
	query.RemoveOneDayRoundTrip = true

	entries, err := h.eventsProvider.GetEvents(ctx, query, 0)

	if err != nil {
		h.logger.Error("Internal error", log.Error(err))
		return nil, err
	}
	return h.buildResponse(entries), nil
}

func (h *PersonalizationService) buildResponse(entries []models.Event) *personalsearchproto.TGetPersonalSearchResponseV2 {
	response := &personalsearchproto.TGetPersonalSearchResponseV2{Entries: make([]*personalsearchproto.TPersonalSearchEntryV2, 0)}
	for _, entry := range entries {
		pbEntry := &personalsearchproto.THotelEntry{
			PointTo: &personalsearchproto.TGeoPoint{
				PointCode: entry.PointTo.PointKey,
				GeoId:     entry.PointTo.GeoID,
			},
			CheckInDate:  entry.When,
			CheckOutDate: entry.ReturnDate,
			Type:         getEventTypeProto(entry),
			Travelers: &personalsearchproto.TTravelers{
				Adults:       int64(entry.Travelers.Adults),
				ChildrenAges: mapChildrenAges(entry.Travelers.ChildrenAges),
			},
		}
		response.Entries = append(
			response.Entries,
			&personalsearchproto.TPersonalSearchEntryV2{Entry: &personalsearchproto.TPersonalSearchEntryV2_Hotel{Hotel: pbEntry}},
		)
	}
	return response
}

func (h *PersonalizationService) GetServiceRegisterer() func(*grpc.Server) {
	return func(server *grpc.Server) {
		personalsearchproto.RegisterPersonalizationServiceV2Server(server, h)
	}
}
