package personalsearch

import (
	"fmt"
	"sort"
	"strconv"
	"strings"
	"time"

	timeformats "cuelang.org/go/pkg/time"

	"a.yandex-team.ru/travel/avia/personalization/internal/consts"
	"a.yandex-team.ru/travel/avia/personalization/internal/services/personalsearch/models"
	"a.yandex-team.ru/travel/library/go/containers"
)

func getNextDate(date string) string {
	parsed, _ := time.Parse(timeformats.RFC3339Date, date)
	if parsed.IsZero() {
		return ""
	}
	return parsed.Add(24 * time.Hour).Format(timeformats.RFC3339Date)
}

func getSortedEvents(events []models.Event, mainService string) []models.Event {
	result := make([]models.Event, 0, len(events))
	result = append(result, events...)

	sort.SliceStable(
		result,
		func(l, r int) bool {
			if result[l].Service == result[r].Service {
				leftIsOrder := result[l].Order.ID != ""
				rightIsOrder := result[r].Order.ID != ""
				if leftIsOrder != rightIsOrder {
					return leftIsOrder
				} else {
					return result[l].Timestamp > result[r].Timestamp
				}
			}
			return result[l].Service != mainService
		},
	)
	return result
}

func buildDirectionKey(event models.Event, query *Query) string {
	if query.TravelService == consts.HotelsServiceName {
		return fmt.Sprintf("hotel_at_%v", event.PointTo.PointKey)
	}
	return fmt.Sprintf("%v-%v", event.PointFrom.PointKey, event.PointTo.PointKey)
}

func validatePoints(event models.Event) bool {
	hasOrigin := event.PointFrom.PointKey != ""
	hasDestination := event.PointTo.PointKey != ""
	isHotelEvent := event.Service == consts.HotelsServiceName
	return hasDestination && (isHotelEvent || hasOrigin)
}

func deduplicateByDirection(events []models.Event, query *Query) []models.Event {
	eventIdxByDirection := make(map[string]int)
	for idx, currentEvent := range events {
		var directionKey = buildDirectionKey(currentEvent, query)
		if prevIdx, ok := eventIdxByDirection[directionKey]; ok {
			if !currentEvent.Order.IsEmpty() && events[prevIdx].Order.IsEmpty() {
				eventIdxByDirection[directionKey] = idx
			}
		} else {
			eventIdxByDirection[directionKey] = idx
		}
	}
	indices := make([]int, 0, len(eventIdxByDirection))
	for _, idx := range eventIdxByDirection {
		indices = append(indices, idx)
	}
	sort.SliceStable(
		indices, func(i, j int) bool {
			return indices[i] < indices[j]
		},
	)
	result := make([]models.Event, 0, len(eventIdxByDirection))
	for _, idx := range indices {
		result = append(result, events[idx])
	}
	return result
}

func removeEventsWithInvalidPoints(events []models.Event) []models.Event {
	result := make([]models.Event, 0, len(events))
	for _, e := range events {
		if !validatePoints(e) {
			continue
		}
		result = append(result, e)
	}
	return result
}

func removeUnconfirmedOrders(events []models.Event) []models.Event {
	confirmedOrderIDs := getConfirmedOrders(events)
	filtered := make([]models.Event, 0, len(events))
	usedOrderIDs := containers.SetOf[string]()
	for _, e := range events {
		if !e.Order.IsEmpty() && (!confirmedOrderIDs.Contains(e.Order.ID) || usedOrderIDs.Contains(e.Order.ID) || !e.Order.IsConfirmed()) {
			continue
		}
		filtered = append(filtered, e)
		if e.Order.ID != "" {
			usedOrderIDs.Add(e.Order.ID)
		}
	}
	return filtered
}

func getConfirmedOrders(events []models.Event) containers.Set[string] {
	eventsCopy := make([]models.Event, 0, len(events))
	eventsCopy = append(eventsCopy, events...)
	sort.SliceStable(
		eventsCopy, func(i, j int) bool {
			return eventsCopy[i].Timestamp < eventsCopy[j].Timestamp
		},
	)
	confirmedOrderIDs := containers.SetOf[string]()
	for _, e := range eventsCopy {
		if e.Order.IsEmpty() {
			continue
		}
		if e.Order.IsConfirmed() {
			confirmedOrderIDs.Add(e.Order.ID)
		} else {
			confirmedOrderIDs.Remove(e.Order.ID)
		}
	}
	return confirmedOrderIDs
}

func fillAviaReturnDate(events []models.Event) []models.Event {
	for i := 0; i < len(events); i++ {
		if events[i].Service != consts.AviaServiceName {
			continue
		}
		if events[i].ReturnDate == "" {
			events[i].ReturnDate = getNextDate(events[i].When)
		}
	}
	return events
}

func removeLongTermAviaOrders(events []models.Event) []models.Event {
	result := make([]models.Event, 0, len(events))
	for _, e := range events {
		if isLongTermAviaOrder(e) {
			continue
		}
		result = append(result, e)
	}
	return result
}

func removeOneDayRoundTrip(events []models.Event) []models.Event {
	result := make([]models.Event, 0, len(events))
	for _, e := range events {
		if isOneDayRoundTrip(e) {
			continue
		}
		result = append(result, e)
	}
	return result
}

func isOneDayRoundTrip(e models.Event) bool {
	if e.Service != consts.AviaServiceName {
		return false
	}
	return e.ReturnDate != "" && e.ReturnDate == e.When
}

func isLongTermAviaOrder(e models.Event) bool {
	if e.Service != consts.AviaServiceName || e.Order.IsEmpty() {
		return false
	}
	const thresholdDays = 30
	return e.ReturnDate != "" && getDatesDiff(e.When, e.ReturnDate) > thresholdDays
}

func getDatesDiff(dateForwardRaw string, dateBackwardRaw string) int {
	dateForward, _ := time.Parse(timeformats.RFC3339Date, dateForwardRaw)
	dateBackward, _ := time.Parse(timeformats.RFC3339Date, dateBackwardRaw)
	return int(dateBackward.Sub(dateForward).Hours() / 24)
}

func filterOutdated(entries []models.Event, now time.Time) []models.Event {
	filtered := make([]models.Event, 0, len(entries))
	today := now.Format(timeformats.RFC3339Date)
	for _, e := range entries {
		if e.When < today {
			continue
		}
		filtered = append(filtered, e)
	}
	return filtered
}

func getChildrenAges(ages string) []int {
	result := make([]int, 0)
	for _, value := range strings.Split(ages, ",") {
		if intValue, err := strconv.Atoi(value); err == nil {
			if intValue < 18 {
				result = append(result, intValue)
			}
		}
	}
	return result
}
