import time
import uuid
from logging import getLogger

from flask import g, Flask, request
from raven.contrib.flask import Sentry

from travel.avia.price_index.lib.rates_provider import rates_provider
from travel.avia.price_index.lib.tvm import tvm_verifier
from travel.avia.price_index.views.dump import dump_view
from travel.avia.price_index.views.helpers import slave_wrapper, jsend_view, bad_request_response
from travel.avia.price_index.views.min_price_batch_search_view import min_price_batch_search_view
from travel.avia.price_index.views.next_days_min_prices_view import next_days_min_prices_view
from travel.avia.price_index.views.search import search_view
from travel.avia.price_index.views.top_direction_prices_by_date_window_view import (
    top_direction_prices_by_date_window_view,
)

application = Flask(__name__)
sentry = Sentry(application)

access_log = getLogger('access')
error_log = getLogger('error')


@application.before_request
def log_request_info_before():
    g.request_start_time = time.time()
    g.session_id = str(uuid.uuid4())
    try:
        g.query_source = tvm_verifier.get_source_by_ticket(request.headers.get('X-Ya-Service-Ticket'))
    except Exception as e:
        g.query_source = None
        error_log.exception('in ticket check: %s', e)
        return bad_request_response('malformed ticket')


@application.after_request
def log_request_info_after(response):
    access_log.info(
        '%s %s %.5fs %s %s',
        request.method,
        response.status,
        time.time() - g.request_start_time,
        request.path,
        str(g.query_source),
        # request.data.decode('utf-8')
    )

    return response


@application.route(
    '/search_methods/v1/search/<national_version>/'
    '<start>/<end>/'
    '<forward_date>/<backward_date>/'
    '<from_id>/<to_id>/'
    '<adults_count>/<children_count>/<infants_count>',
    methods=['POST'],
    endpoint='search_methods_two_ways',
)
@application.route(
    '/search_methods/v1/search/<national_version>/'
    '<start>/<end>/'
    '<forward_date>/'
    '<from_id>/<to_id>/'
    '<adults_count>/<children_count>/<infants_count>',
    methods=['POST'],
    endpoint='search_methods_one_ways',
)
@slave_wrapper
@jsend_view
def v3_search(session, *args, **kwargs):
    search_form = search_view.parse_form(kwargs, request.data, g.query_source)
    return search_view.process(session, search_form)


@application.route('/search_methods/v1/dump/', methods=['GET'], endpoint='dump')
@slave_wrapper
@jsend_view
def dump(session, *args, **kwargs):
    form = dump_view.parse_form(request.args)
    return dump_view.process(session, form)


@application.route(
    '/search_methods/v1/min_price_batch_search/<national_version>', methods=['POST'], endpoint='min_price_batch_search'
)
@slave_wrapper
@jsend_view
def min_price_batch_search(session, *args, **kwargs):
    form = min_price_batch_search_view.parse_form(kwargs, request.data, g.query_source)
    return min_price_batch_search_view.process(session, form)


@application.route('/search_methods/v1/top_directions_by_date_window/<national_version>', methods=['post'])
def top_directions_by_window_date_search(**kwargs):
    return top_direction_prices_by_date_window_view.process(kwargs, request.data, g.query_source)


@application.route('/search_methods/v1/next_days_min_prices/<national_version>', methods=['post'])
def next_days_min_prices_search(**kwargs):
    return next_days_min_prices_view.process(kwargs, request.data, g.query_source)


@application.route('/ping', methods=['GET'])
def ping():
    if application.shutdown_flag.is_set():
        return 'Stopped', 410

    return 'ok', 200


@application.route('/shutdown', methods=['POST'])
def shutdown():
    try:
        application.shutdown_flag.set()
    except Exception:
        application.exception('Can not stop ping responses')
        return 'error', 500
    else:
        return 'ok', 200


if __name__ == '__main__':
    rates_provider.fetch()
    application.run(host='0.0.0.0', port=80)
