# -*- coding: utf-8 -*-
import logging
from argparse import ArgumentParser
from datetime import datetime, timedelta

import sys
from enum import Enum

from travel.avia.library.python.lib_yt.client import configured_client
from travel.avia.library.python.urls import travel_avia_search, Environment as UrlEnvironment
from travel.avia.price_index.bin.ad_feed_dump.dumper import Handler
from travel.avia.price_index.lib.settings import settings

logger = logging.getLogger(__name__)


class Environment(Enum):
    TESTING = 'testing'
    PRODUCTION = 'production'

    def __str__(self):
        return self.value


ENV_TO_URL_ENV = {
    Environment.TESTING: UrlEnvironment.TESTING,
    Environment.PRODUCTION: UrlEnvironment.PRODUCTION,
}

OUTPUT_TABLE_FOR_ENVIRONMENT = {
    # Templates for output table. nv = national version
    Environment.TESTING: '//home/avia/testing/data/price-index/{nv}/ad-feed',
    Environment.PRODUCTION: '//home/avia/data/price-index/{nv}/ad-feed',
}

LANDING_ROUTES_TABLE = '//home/avia/avia-statistics/landing-routes'

if __name__ == '__main__':
    logging.basicConfig(
        level=logging.INFO,
        stream=sys.stdout,
    )

    parser = ArgumentParser()

    parser.add_argument('--national-version', required=True, type=str, choices=('ru', 'ua', 'kz', 'com', 'tr'))
    parser.add_argument('--update-age', default=30, type=int, help='Maximum allowed price update age (days)')
    parser.add_argument('--days', default=30, type=int, help='Maximum number of days to get prices from future')
    parser.add_argument('--max-transfers', default=2, type=int)

    subparser = parser.add_subparsers(
        title='mode',
        dest='mode',
        help='Automatically or manually fill table names. '
        'Auto mode is convenient for usage in a particular environment. '
        'Manual mode is useful for debugging and development.',
        required=True,
    )

    parser_auto = subparser.add_parser('auto')
    parser_auto.add_argument('environment', type=Environment, choices=list(Environment))

    parser_manual = subparser.add_parser('manual')
    parser_manual.add_argument(
        '--landing-routes-table',
        required=True,
        type=str,
        help='Path to landing routes table in YT. Used to filter invalid routes',
    )
    parser_manual.add_argument('--output-table', required=True, type=str)
    parser_manual.add_argument('--travel-host', required=True, type=str)

    arguments = parser.parse_args()
    if 'auto' == arguments.mode:
        landing_routes_table = LANDING_ROUTES_TABLE
        output_table = OUTPUT_TABLE_FOR_ENVIRONMENT[arguments.environment].format(nv=arguments.national_version)
        host: str = travel_avia_search(ENV_TO_URL_ENV[arguments.environment]).travel_host
    else:
        landing_routes_table = arguments.landing_routes_table
        output_table = arguments.output_table
        host: str = arguments.travel_host

    yt_client = configured_client('hahn', settings.YT_TOKEN)
    tomorrow = datetime.now() + timedelta(days=1)

    handler = Handler(yt_client=yt_client)
    handler.create_feed(
        from_date=tomorrow,
        to_date=tomorrow + timedelta(days=arguments.days),
        updated_at=datetime.now() - timedelta(days=arguments.update_age),
        max_transfers=arguments.max_transfers,
        destination_table=output_table,
        national_version=arguments.national_version,
        landing_routes_table=landing_routes_table,
        host=host,
    )
