import logging
from logging import Logger

import psycopg2.extras
from sqlalchemy import create_engine
from sqlalchemy.orm import sessionmaker

from travel.avia.price_index.lib.db.timeout_proxy import TimeOutProxy
from travel.avia.price_index.lib.settings import settings


class Storage(object):
    def __init__(self, connection_string, enable_log_sql_queries, sql_log):
        # type: (str, bool, Logger) -> None
        if enable_log_sql_queries:
            self._engine = self._create_logging_engine(connection_string)
        else:
            self._engine = self._create_simple_engine(connection_string)
        self._session_fabric = sessionmaker(bind=self._engine)
        self._sql_log = sql_log

    def get_session(self):
        return self._session_fabric()

    def get_engine(self):
        return self._engine

    def _create_logging_engine(self, connection_string):
        def _connection_factory(*args, **kwargs):
            connection = psycopg2.extras.LoggingConnection(*args, **kwargs)
            connection.initialize(self._sql_log)
            return connection

        return create_engine(
            connection_string,
            proxy=TimeOutProxy(settings.PGAAS_STATEMENT_TIMEOUT),
            connect_args={
                'connection_factory': _connection_factory,
            },
        )

    def _create_simple_engine(self, connection_string):
        return create_engine(
            connection_string,
            proxy=TimeOutProxy(settings.PGAAS_STATEMENT_TIMEOUT),
            echo=False,
        )


storage = Storage(
    connection_string=settings.MASTER_CONNECTION_STRING_FOR_WORK,
    enable_log_sql_queries=settings.ENABLE_LOG_SQL_QUERIES,
    sql_log=logging.getLogger('sql'),
)

slave_storage = Storage(
    connection_string=settings.SLAVE_CONNECTION_STRING,
    enable_log_sql_queries=settings.ENABLE_LOG_SQL_QUERIES,
    sql_log=logging.getLogger('sql'),
)
