from typing import Tuple

from more_itertools import pairwise

from travel.avia.price_index.models.flight import Flight
from travel.avia.price_index.models.index_data import TransferIndex


class TransferIndexBuilder(object):
    def index(self, departure_flights, arrival_flights):
        # type: (Tuple[Flight, ...], Tuple[Flight, ...]) -> TransferIndex
        return TransferIndex(
            count=self._get_count_transfer(departure_flights, arrival_flights),
            duration=self._get_min_transfer_duration(departure_flights, arrival_flights),
            has_night_transfer=self._has_night_transfer(departure_flights, arrival_flights),
            has_airport_change=self._has_airport_change(departure_flights, arrival_flights),
        )

    def _get_count_transfer(self, forward_flights, backward_flights):
        forward = len(forward_flights)
        backward = len(backward_flights)

        forward_transfer = forward - 1 if forward > 1 else 0
        backward_transfer = backward - 1 if backward > 1 else 0

        return max(forward_transfer, backward_transfer)

    def _get_min_transfer_duration(self, forward_flights, backward_flights):
        forward_transfer_time = self._get_min_transfer_duration_for_direction(forward_flights)
        backward_transfer_time = self._get_min_transfer_duration_for_direction(backward_flights)

        if forward_transfer_time is None and backward_transfer_time is None:
            return None

        if forward_transfer_time is None:
            return backward_transfer_time

        if backward_transfer_time is None:
            return forward_transfer_time

        return min(forward_transfer_time, backward_transfer_time)

    def _get_min_transfer_duration_for_direction(self, flights):
        if len(flights) < 2:
            return None

        transfer_time = float("inf")
        for i in range(1, len(flights)):
            start = flights[i - 1]
            end = flights[i]

            current_transfer = (end.departure_utc - start.arrival_utc).seconds / 60

            transfer_time = min(transfer_time, current_transfer)

        return int(transfer_time)

    def _has_night_transfer(self, forward_flights, backward_flights):
        return self._has_night_transfer_for_direction(forward_flights) or self._has_night_transfer_for_direction(
            backward_flights
        )

    def _has_night_transfer_for_direction(self, flights):
        if len(flights) < 2:
            return False

        for prev_flight, next_flight in pairwise(flights):
            prev_arrival = prev_flight.arrival
            next_departure = next_flight.departure

            if 0 <= prev_arrival.hour < 6 or 0 <= next_departure.hour < 6 or prev_arrival.day != next_departure.day:
                return True

        return False

    def _has_airport_change(self, forward_flights, backward_flights):
        return self._has_airport_change_for_direction(forward_flights) or self._has_airport_change_for_direction(
            backward_flights
        )

    def _has_airport_change_for_direction(self, flights):
        if len(flights) < 2:
            return False

        for i in range(1, len(flights)):
            if flights[i - 1].to_id != flights[i].from_id:
                return True

        return False


transfer_index_builder = TransferIndexBuilder()
