import os
import sys
import logging
from logging.config import dictConfig

from travel.avia.price_index.lib.flask_logger import FlaskLogger
from travel.avia.price_index.lib.settings import settings


def setup_logs(write_to_stdout=False):
    log_path = settings.LOG_ROOT

    root_handlers = ['error', 'info']
    if settings.DEBUG:
        root_handlers.append('debug')

    if write_to_stdout:
        root_handlers.append('stdout')

    logging_config = dict(
        version=1,
        formatters={
            'verbose': {'format': '%(levelname)s %(asctime)s: %(message)s'},
            'message_only': {'format': '%(asctime)s: %(message)s'},
            'sql_command': {'format': '%(asctime)s:>>> %(message)s;'},
            'error': {
                'format': '%(levelname)s %(asctime)s %(process)d ' '%(pathname)s:%(funcName)s:%(lineno)d %(message)s',
            },
            'json': {'format': '%(message)s', '()': 'pythonjsonlogger.jsonlogger.JsonFormatter'},
        },
        handlers={
            'stdout': {
                'level': 'INFO',
                'class': 'logging.StreamHandler',
                'stream': sys.stdout,
                'formatter': 'verbose',
            },
            'error': {
                'class': 'logging.handlers.WatchedFileHandler',
                'filename': os.path.join(log_path, 'error.log'),
                'formatter': 'error',
                'level': 'ERROR',
            },
            'info': {
                'class': 'logging.handlers.WatchedFileHandler',
                'filename': os.path.join(log_path, 'info.log'),
                'formatter': 'verbose',
                'level': 'INFO',
            },
            'access': {
                'class': 'logging.handlers.WatchedFileHandler',
                'filename': os.path.join(log_path, 'access.log'),
                'formatter': 'verbose',
                'level': 'INFO',
            },
            'yt.price_precision': {
                'class': 'logging.handlers.WatchedFileHandler',
                'filename': os.path.join(log_path, 'yt/price_precision.log'),
                'formatter': 'json',
                'level': 'INFO',
            },
            'debug': {
                'class': 'logging.StreamHandler',
                'formatter': 'message_only',
                'level': 'DEBUG',
            },
            'sql': {
                'class': 'logging.handlers.WatchedFileHandler',
                'filename': os.path.join(log_path, 'sql.log'),
                'formatter': 'sql_command',
                'level': 'DEBUG',
            },
            'sentry': {
                'level': 'ERROR',
                'class': 'raven.handlers.logging.SentryHandler',
            },
        },
        root={
            'handlers': root_handlers,
            'level': logging.DEBUG,
        },
        loggers={
            'access': {
                'handlers': ['access'],
                'propagate': False,
                'level': 'INFO',
            },
            'error': {
                'handlers': ['error', 'sentry'],
                'propagate': False,
                'level': 'ERROR',
            },
            'yt.price_precision': {
                'handlers': ['yt.price_precision'],
                'level': 'INFO',
                'propagate': False,
            },
            'sql': {
                'handlers': ['sql'],
                'level': 'DEBUG',
                'propagate': False,
            },
        },
    )

    dictConfig(logging_config)
    logging.setLoggerClass(FlaskLogger)
