import time
from logging import Logger, getLogger

from typing import Optional

from travel.avia.price_index.models.filters import Filters


class Context(object):
    def __init__(self, yt_logger, query_source=None):
        # type: (Logger, Optional[int]) -> None
        self._yt_logger = yt_logger
        self._query_source = query_source
        self._data = {
            'unixtime': int(time.time()),
            'fromId': None,
            'toId': None,
            'forwardDate': None,
            'backwardDate': None,
            'hasPrice': None,
            'nationalVersionId': None,
            'withBaggage': None,
            'airlines': None,
            'transferCount': None,
            'transferMinDuration': None,
            'transferMaxDuration': None,
            'transferHasAirportChange': None,
            'transferHasNight': None,
            'timeForwardArrival': None,
            'timeForwardDeparture': None,
            'timeBackwardArrival': None,
            'timeBackwardDeparture': None,
            'airportsForwardDeparture': None,
            'airportsForwardArrival': None,
            'airportsForwardTransfers': None,
            'airportsBackwardDeparture': None,
            'airportsBackwardArrival': None,
            'airportsBackwardTransfers': None,
        }

    def log(
        self,
        from_id,
        to_id,
        forward_date,
        backward_date,
        adults_count,
        children_count,
        infants_count,
        national_version_id,
        has_price,
    ):
        # type: (int, int, str, Optional[str], int, int, int, int, bool) -> None
        item = dict(self._data, querySource=self._query_source)

        item['fromId'] = from_id
        item['toId'] = to_id
        item['forwardDate'] = forward_date
        item['backwardDate'] = backward_date
        item['nationalVersionId'] = national_version_id
        item['hasPrice'] = has_price
        item['adults_count'] = adults_count
        item['children_count'] = children_count
        item['infants_count'] = infants_count

        self._yt_logger.info(item)

    @staticmethod
    def _serialize_list(array):
        return ','.join(str(i) for i in sorted(array)) if array else None

    def fill_filter_data(self, filter_model):
        # type: (Filters) -> Context
        item = self._data
        item['withBaggage'] = filter_model.with_baggage
        item['airlines'] = self._serialize_list(filter_model.airlines)
        item['transferCount'] = filter_model.transfer_filters.count
        item['transferMinDuration'] = filter_model.transfer_filters.min_duration
        item['transferMaxDuration'] = filter_model.transfer_filters.max_duration
        item['transferHasAirportChange'] = filter_model.transfer_filters.has_airport_change
        item['transferHasNight'] = filter_model.transfer_filters.has_night
        item['timeForwardArrival'] = self._serialize_list(filter_model.time_filters.forward_arrival)
        item['timeForwardDeparture'] = self._serialize_list(filter_model.time_filters.forward_departure)
        item['timeBackwardArrival'] = self._serialize_list(filter_model.time_filters.backward_arrival)
        item['timeBackwardDeparture'] = self._serialize_list(filter_model.time_filters.backward_departure)
        item['airportsForwardDeparture'] = self._serialize_list(filter_model.airports_filters.forward_departure)
        item['airportsForwardArrival'] = self._serialize_list(filter_model.airports_filters.forward_arrival)
        item['airportsForwardTransfers'] = self._serialize_list(filter_model.airports_filters.forward_transfers)
        item['airportsBackwardDeparture'] = self._serialize_list(filter_model.airports_filters.backward_departure)
        item['airportsBackwardArrival'] = self._serialize_list(filter_model.airports_filters.backward_arrival)
        item['airportsBackwardTransfers'] = self._serialize_list(filter_model.airports_filters.backward_transfers)

        return self


class PricePrecisionLogger(object):
    def __init__(self, logger, yt_logger):
        # type: (Logger, Logger) -> None
        self._logger = logger
        self._yt_logger = yt_logger

    def start_log(self, query_source=None):
        return Context(yt_logger=self._yt_logger, query_source=query_source)


price_precision_logger = PricePrecisionLogger(logger=getLogger(__name__), yt_logger=getLogger('yt.price_precision'))
