# -*- coding: utf-8 -*-


from typing import Tuple, Dict

from sqlalchemy import and_, or_
from sqlalchemy.orm import Session, Query as SaQuery

from travel.avia.price_index.db_models.result import Result
from travel.avia.price_index.lib.constants import NULL_DATE
from travel.avia.price_index.lib.search_query_finder import History
from travel.avia.price_index.models.query import Query


class FastMinQuerySearcher(object):
    def batch_find(self, session, queries):
        # type: (Session, Tuple[Query, ...]) -> Tuple[Dict[Query, int], Dict[Query, History]]

        if not queries:
            return {}, {}

        key_to_query = {
            (
                q.national_version_id,
                q.from_id,
                q.to_id,
                q.forward_date,
                q.backward_date,
                q.adults_count,
                q.children_count,
                q.infants_count,
            ): q
            for q in queries
        }

        q = self._sql(session, queries)
        base_value_by_query = {}
        history_by_query = {}

        for (
            national_version_id,
            from_id,
            to_id,
            forward_date,
            backward_date,
            adults_count,
            children_count,
            infants_count,
            base_value,
            updated_at,
        ) in session.execute(q).fetchall():
            q = key_to_query[
                (
                    national_version_id,
                    from_id,
                    to_id,
                    forward_date,
                    None if backward_date == NULL_DATE else backward_date,
                    adults_count,
                    children_count,
                    infants_count,
                )
            ]

            base_value_by_query[q] = base_value
            history_by_query[q] = History(updated_at, base_value is not None, requested_query=q)

        return base_value_by_query, history_by_query

    def _sql(self, session, queries):
        # type: (Session, Tuple[Query, ...]) -> SaQuery
        return session.query(
            Result.national_version_id,
            Result.from_id,
            Result.to_id,
            Result.forward_date,
            Result.backward_date,
            Result.adults_count,
            Result.children_count,
            Result.infants_count,
            Result.base_value,
            Result.updated_at,
        ).filter(
            or_(
                *[
                    and_(
                        Result.national_version_id == query.national_version_id,
                        Result.adults_count == query.adults_count,
                        Result.children_count == query.children_count,
                        Result.infants_count == query.infants_count,
                        Result.from_id == query.from_id,
                        Result.to_id == query.to_id,
                        Result.forward_date == query.forward_date,
                        Result.backward_date == (query.backward_date or NULL_DATE),
                    )
                    for query in queries
                ]
            )
        )


searcher = FastMinQuerySearcher()
